/**
 * Copyright 2021 NXP
 *
 * This software is owned or controlled by NXP and may only be used
 * strictly in accordance with the applicable license terms.  By expressly
 * accepting such terms or by downloading, installing, activating and/or
 * otherwise using the software, you are agreeing that you have read, and
 * that you agree to comply with and are bound by, such license terms.  If
 * you do not agree to be bound by the applicable license terms, then you
 * may not retain, install, activate or otherwise use the software.
 *
 */

/** @file
 *
 * alpar.h:
 *
 *
 * Project:  PN7462AU
 *
 * $Date: 2021-09-23 15:00:00 +0200 (Thu, 23 Sep 2021) $
 * $Author: Petar Hlad (MobileKnowledge) $
 * $Revision:
 */

#ifndef ALPAR_H_
#define ALPAR_H_

/**
 * Includes
 */

#include <stdint.h>
#include "ph_Status.h"
#include "phhalCt.h"
#include "phpalCt.h"


/* *****************************************************************************************************************
 * Internal Definitions
 * ***************************************************************************************************************** */

#define PH_EXCTEMVCO_MAX_ATR_SIZE           32
#define PH_EXCTEMVCO_MAX_NO_APDU             9
#define PH_EXCTEMVCO_APDU_LENGTH             13

#define PH_EX_WAKEUP_GPIO_PIN           0x0006U
#define PH_ENABLE                       0x0001U /**< Enabled. */
#define PH_DISABLE                      0x0000U /**< Disabled. */

#define PH_EX_HSU_BUFFER_SIZE           400U  //   1k buffer

/**
 * General ALPAR command values.
 */

#define ALPAR_CMD_CARD_COMMAND          0x00U /** Sends an APDU to the activated smart card. */
#define ALPAR_CMD_CHECK_PRES_CARD       0x09U /** Check the selected card presence. */
#define ALPAR_CMD_GET_FW_VERSION        0x0AU /** Reads the firmware version. */
#define ALPAR_CMD_SET_CARD_BAUD_RATE    0x0BU /** Sets the baudrate of the activated smart card. */
#define ALPAR_CMD_IFSD_REQUEST          0x0CU /** Request an IFSD change for the activated smart card communication. */
#define ALPAR_CMD_SET_SERIAL_BAUD_RATE  0x0DU /** Changes the baudrate for host communication. */
#define ALPAR_CMD_SHOW_FIDI             0x0EU /** Displays the current FiDi. */
#define ALPAR_CMD_SET_SERIAL_TIMEOUT    0x0FU /** Sets the timeout for the serial connection. */
#define ALPAR_CMD_NEGOTIATE_PPS         0x10U /** Initiates a parameter change for t=0. */
#define ALPAR_CMD_SET_CLOCK_CARD        0x11U /** Selects the division for the smart card clock. */
#define ALPAR_CMD_TDA_WRITE_I2C         0x2BU /** Write to a TDA connected to the CPU through I2C interface */
#define ALPAR_CMD_TDA_READ_I2C          0x2CU /** Reads from a TDA connected to the CPU through I2C interface */
#define ALPAR_CMD_START_EMV_LOOPBACK    0x2FU /** Launch the EMV loopback process. Blocking function that does not return. */
#define ALPAR_CMD_POWER_OFF             0x4DU /** Deactivates the current smart card. */
#define ALPAR_CMD_SELECT_DEVICE         0x67U /** Selects the connected TDA daughter board. */
#define ALPAR_CMD_POWER_UP_1V8          0x68U /** Activates the card with VCC=1.8V. */
#define ALPAR_CMD_SELECT_CARD           0x6AU /** Selects the smart card to access. */
#define ALPAR_CMD_POWER_UP_3V           0x6DU /** Activates the card with VCC=3V. */
#define ALPAR_CMD_POWER_UP_5V           0x6EU /** Activates the card with VCC=5V. */
#define ALPAR_CMD_SET_NAD               0xA5U /** Sets the NAD parameter for T=1 communication. */
#define ALPAR_CMD_IDLE_MODE             0xA9U /** Sets the smart card in idle mode (activated with lower consumption). */
#define ALPAR_CMD_GET_READER_STATUS     0xAAU /** Displays information about the current state of the reader. */

#define ALPAR_CMD_BAD_OPCODE            0xFFU

/**
 * ALPAR command, only outgoing.
 */

#define ALPAR_OUTGOING_CMD_CARD_PRES    0xA0 /** Sent to host when a card insertion or extraction is detected **/

/**
 * ALPAR Errors
 */

#define ALPAR_ERR_SUCCESS               0x00U
#define ALPAR_ERR_DATA_LENGTH_TOO_SHORT 0x08U
#define ALPAR_ERR_WRONG_APDU            0x20U
#define ALPAR_ERR_TOO_SHORT_APDU        0x21U
#define ALPAR_ERR_CARD_MUTE_T1_EXG      0x22U
#define ALPAR_ERR_BAD_NAD               0x24U
#define ALPAR_ERR_BAD_LRC               0x25U
#define ALPAR_ERR_RESYNC                0x26U
#define ALPAR_ERR_CHAIN_ABORTED         0x27U
#define ALPAR_ERR_CARD_OVERFLOW         0x29U
#define ALPAR_ERR_NON_NEGOTIABLE_MODE   0x30U
#define ALPAR_ERR_UNKNOWN_PROTOCOL      0x31U
#define ALPAR_ERR_T1_NOT_ACCEPTED       0x32U
#define ALPAR_ERR_PPS_BAD_ANSWER        0x33U
#define ALPAR_ERR_PCK_ERROR             0x34U
#define ALPAR_ERR_BAD_PARAMETER         0x35U
#define ALPAR_ERR_TB3_ABSENT            0x38U
#define ALPAR_ERR_PPS_NOT_ACCEPTED      0x39U
#define ALPAR_ERR_EARLY_ANSWER          0x3BU
#define ALPAR_ERR_CARD_DEACTIVATED      0x40U
#define ALPAR_ERR_UNKNOWN_COMMAND       0x55U
#define ALPAR_ERR_CARD_MUTE_PWR_ON      0x80U
#define ALPAR_ERR_TIMEOUT               0x81U
#define ALPAR_ERR_PARITY_ERR_RX         0x83U
#define ALPAR_ERR_PARITY_ERR_TX         0x84U
#define ALPAR_ERR_BAD_FIDI              0x86U
#define ALPAR_ERR_EXCEEDED_ATR_DURATION 0x88U
#define ALPAR_ERR_CWI_NOT_SUPPORTED     0x89U
#define ALPAR_ERR_BWI_NOT_SUPPORTED     0x8AU
#define ALPAR_ERR_WI_NOT_SUPPORTED      0x8BU
#define ALPAR_ERR_TC3_NOT_ACCEPTED      0x8CU
#define ALPAR_ERR_ATR_PARITY_ERR        0x8DU
#define ALPAR_ERR_SPECIFIC_MODE_BIT_ERR 0x92U
#define ALPAR_ERR_TB1_ABSENT            0x93U
#define ALPAR_ERR_TB1_NONZERO           0x94U
#define ALPAR_ERR_IFSC_WRONG_VALUE      0x95U
#define ALPAR_ERR_WRONG_TDI             0x96U
#define ALPAR_ERR_TB2_PRESENT           0x97U
#define ALPAR_ERR_TC1_NON_COMPATIBLE    0x98U
#define ALPAR_ERR_IFSD_NOT_ACCEPTED     0x99U
#define ALPAR_ERR_NOT_T1_CARD           0X9BU
#define ALPAR_ERR_PROCEDURE_BYTE_ERR    0xA0U
#define ALPAR_ERR_CARD_ABSENT           0xC0U
#define ALPAR_ERR_CHECKSUM_ERR          0xC3U
#define ALPAR_ERR_ATR_NOT_SUPPORTED     0xC6U
#define ALPAR_ERR_CARD_ERR              0xE0U
#define ALPAR_ERR_CLK_FREQ_NOT_ACCEPTED 0xE1U
#define ALPAR_ERR_UART_OVERFLOW         0xE2U
#define ALPAR_ERR_SPLY_VOLTAGE_DROPOFF  0xE3U
#define ALPAR_ERR_TEMPERATURE_ALARM     0xE4U
#define ALPAR_ERR_FRAMING_ERR           0xE9U
#define ALPAR_ERR_SERIAL_LCR_ERR        0xF0U
#define ALPAR_ERR_CMD_FRAME_LOST        0xF1U
#define ALPAR_ERR_SERIAL_TIMEOUT        0xFFU

/**
 * Structs
 */

typedef uint8_t aStatus_t;

/**
 * Extern variables
 */

extern uint8_t gphhalCt_BCurrentFiDi;
extern uint8_t gphEx_SysHsuRxBuffer[];
extern uint8_t ATR[100];

/**
 * ALPAR Command functions
 */

/**
 * @brief This function handles the ALPAR command received from the host and calls the corresponding Ct function.
 * @param[in]   cmd         ALPAR command
 * @param[in]   data        Command data
 * @param[in]   datalength  Length of the command data
 * @return #PH_ERR_SUCCESS is always returned
 */
phStatus_t alpar_handle_command(uint8_t cmd, uint8_t *data, uint16_t datalength);


/**
 * @brief This function is used to send an APDU to the activated smart card.
 * @param[in]   cmd_buff    Buffer containing the card command data
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer containing the response bytes
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return Returns the status of transceiving the command to the card
 */
phStatus_t alpar_card_command(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to check the presence of a card in the main card slot.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response byte containing the card presence
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return #ALPAR_ERR_SUCCESS is always returned
 */
phStatus_t alpar_check_pres_card(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to launch the EMV loopback process. It is a blocking function that does not return.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response bytes.
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return This function does not return.
 */
phStatus_t alpar_start_EMV_loopback(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to power off the current smartcard.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response bytes. For this function it is ignored
 * @param[out]  resp_len    Length of the buffer with the response bytes. For this function it is ignored
 * @return #ALPAR_ERR_SUCCESS is always returned
 */
phStatus_t alpar_power_off(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to activate the card with VCC=5V
 * @param[in]   cmd_buff    Buffer with the command data containing the parameter to select EMV specification compliance
 *                          When set to 0 all the parameters of the ATR compliant with ISO7816-3 will be taken into account.
 *                          When set to 1 only the ATR of cards whose parameters are inside the E.M.V 4.3 specification scope
 *                          will be taken into account
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer with the response byte containing the ATR bytes
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return      Status code from calling alpar_PowerUp for card activation.
 */
phStatus_t alpar_power_up_5V(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to identify the software version.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response bytes containing the FW version coded in ASCII characters
 * @param[out]  resp_len    Length of the buffer with the response bytes
 * @return #ALPAR_ERR_SUCCESS is always returned
 */
phStatus_t alpar_get_fw_version(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief  This function is used to set the baud rate for the main card.
 * @param[in]   cmd_buff    Buffer with the command data byte containing Fi and Di value which are used to
 *                          calculate PDR register values and Clock divider values.
 * @param[in]   cmd_len     Length of command data buffer.
 * @param[out]  resp_buff   Buffer with the response byte containing the current FiFiDi
 * @param[out]  resp_len    Length of the buffer with the response bytes
 * @return    Status code for Baud Rate Setting.
 *
 * @retval    #ALPAR_ERR_SUCCESS   Setting the baud rate is successful.
 * @maskedret #ALPAR_ERR_BAD_FIDI  Parameters are invalid.
 */
phStatus_t alpar_set_card_baud_rate(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used for changing the baud rate onto the serial link between the system and the PN7462.
 * @param[in]   cmd_buff    Buffer with the command data containing the baud rate parameter
 * @param[in]   cmd_len     Length of command data buffer.
 * @param[out]  resp_buff   Buffer with the response bytes. For this function it is ignored
 * @param[out]  resp_len    Length of the buffer with the response bytes. For this function it is ignored
 * @retval    #ALPAR_ERR_SUCCESS   Serial baud rate is changed successfully.
 * @maskedret #ALPAR_ERR_BAD_PARAMETER   Parameters are invalid.
 */
phStatus_t alpar_set_serial_baud_rate(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to retrieve the current FiDi ot the card in use.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response byte containing the current FiFiDi
 * @param[out]  resp_len    Length of the buffer with the response bytes
 * @return #ALPAR_ERR_SUCCESS is always returned
 */
phStatus_t alpar_show_fidi(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used for PPS request and PPS response handling in negotiable mode.
 *        This function also applies new baud rate and new protocol according to negotiated values.
 *
 * @param[in]   cmd_buff    Buffer with the command data containing the parameters for PPS Exchange.
 *                              - FiDi given by TA1 parameter of the ATR
 *                              - NegotiableProtocol to be used for PPS exchange request,
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer with the response bytes. For this function it is ignored
 * @param[out]  resp_len    Length of the buffer with the response bytes. For this function it is ignored
 *
 * @return Status code for PPS Exchange operation.
 *
 * @retval    #ALPAR_ERR_SUCCESS  If Correct PPS Response has been received successfully,structure parameters
 *                                will be updated as per PPS Response .
 * @maskedret #ALPAR_ERR_BAD_PARAMETER  Parameters are invalid.
 * @maskedret #ALPAR_ERR_PPS_BAD_ANSWER    If Wrong PPS response has been received.
 * @maskedret #ALPAR_ERR_PPS_NOT_ACCEPTED  If the PPS exchange did not succeed.
 */
phStatus_t alpar_negotiate_pps(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used for changing the card clock frequency.
 * @param[in]   cmd_buff    Buffer with the command data containing the clock frequency parameter
 * @param[in]   cmd_len     Length of command data buffer.
 * @param[out]  resp_buff   Buffer with the response bytes. For this function it is ignored
 * @param[out]  resp_len    Length of the buffer with the response bytes. For this function it is ignored
 * @retval    #ALPAR_ERR_SUCCESS   Serial baud rate is changed successfully.
 * @maskedret #ALPAR_ERR_CLK_FREQ_NOT_ACCEPTED The provided frequency is incompatible with the current Fi used by the card.
 * @maskedret #ALPAR_ERR_BAD_PARAMETER   Parameters are invalid.
 */
phStatus_t alpar_set_clock_card(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to activate the card with VCC=1.8V
 * @param[in]   cmd_buff    Buffer with the command data containing the parameter to select EMV specification compliance
 *                          When set to 0 all the parameters of the ATR compliant with ISO7816-3 will be taken into account.
 *                          When set to 1 only the ATR of cards whose parameters are inside the E.M.V 4.3 specification scope
 *                          will be taken into account
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer with the response byte containing the ATR bytes
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return      Status code from calling alpar_PowerUp for card activation.
 */
phStatus_t alpar_power_up_1V8(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to activate the card with VCC=3V
 * @param[in]   cmd_buff    Buffer with the command data containing the parameter to select EMV specification compliance
 *                          When set to 0 all the parameters of the ATR compliant with ISO7816-3 will be taken into account.
 *                          When set to 1 only the ATR of cards whose parameters are inside the E.M.V 4.3 specification scope
 *                          will be taken into account
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer with the response byte containing the ATR bytes
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return      Status code from calling alpar_PowerUp for card activation.
 */
phStatus_t alpar_power_up_3V(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * This function is used to set NAD byte for T=1 communication for 7816.
 * @param[in]   cmd_buff    Buffer with the command data containing the parameter value for setting NAD for
 *                          terminal to card communication,bit 8 and 4 should be set as 0.
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer with the response bytes. For this function it is ignored
 * @param[out]  resp_len    Length of the buffer with the response bytes. For this function it is ignored
 *
 * @retval  #ALPAR_ERR_SUCCESS   If NAD is set successfully.
 * @maskedret #ALPAR_ERR_BAD_NAD   If DAD and SAD both have same nonzero value.
 *                                              If bit 8 or bit 4 is set to 1.
 */
phStatus_t alpar_set_nad(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * This function starts or stops the clock according to the ISO7816 feature.
 * @param[in]   cmd_buff    Buffer with the command data containing the two parameters for this function:
 *                              ClockStop: When set to 1, stops the clock. When set to 0, enables the card clock
 *                              LowHigh:    When set to 1, the clock will stop at HIGH position.
 *                                          When set to 0, the clock will stop at LOW.
 * @param[in]   cmd_len     Length of command data buffer
 * @param[out]  resp_buff   Buffer with the response bytes. For this function it is ignored
 * @param[out]  resp_len    Length of the buffer with the response bytes. For this function it is ignored
 *
 * @retval    #ALPAR_ERR_SUCCESS   Clock is stopped or started successfully.
 * @maskedret #ALPAR_ERR_BAD_PARAMETER   Parameters are invalid.
 */
phStatus_t alpar_idle_mode(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used to check the status of the reader.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response byte containing the card presence
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return #ALPAR_ERR_SUCCESS is always returned
 */
phStatus_t alpar_get_reader_status(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is called when a card is inserted or extracted from the slot and is used
 *        to retreive the presence of the card.
 * @param[in]   cmd_buff    Buffer with the command data. For this function it is ignored
 * @param[in]   cmd_len     Length of command data buffer. For this function it is ignored
 * @param[out]  resp_buff   Buffer with the response byte containing the card presence
 * @param[out]  resp_len    Length of the buffer containing the response bytes
 * @return #ALPAR_ERR_SUCCESS is always returned
 */
phStatus_t alpar_outgoing_card_presence(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len);


/**
 * @brief This function is used for processing activation process further after successful card activation.
 *
 * @param[in]   pDataParams     Pointer to the context structure of the PAL layer
 * @param[in]   psAtrPalParams  Pointer to atr parameter structure which will be used in PAL.
 * @return Status of the activation process
 */
phStatus_t alpar_processActivation(phpalCt_DataParams_t *pDataParams, phhalCt_ProtocolParams_t *psAtrPalParams);


/**
 * @brief This Function does a cold reset/activate of the card for ATR reception.
 *
 * @param[in]    bVccSel        Card Vcc Selection [#PHHAL_CT_VCC1M8 :1.8v,#PHHAL_CT_VCC3: 3v or #PHHAL_CT_VCC5: 5v]
 * @param[in]    bEmvEn         Flag for EMVCo profile
 * @param[out]   pbAtrBuffer    Pointer to the ATR buffer where the ATR bytes will be stored.Atr Buffer should not be
 *                              less than 32 bytes for EMVCo and 33 bytes for 7816.
 * @param[out]   pbAtrSize   Pointer to which the count of ATR bytes is copied
 * @return      Status code for card activation.
 *
 */
phStatus_t alpar_PowerUp(uint8_t bVccSel, uint8_t bEmvEn, uint8_t* pbAtrBuffer, uint8_t* pbAtrSize);

#endif /* ALPAR_H_ */
