/*----------------------------------------------------------------------------*/
/* Copyright 2016,2020,2022-2024 NXP                                          */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** \file
 * Global data types.
 * $Author: NXP $
 * $Revision: $ (v04.18.00)
 * $Date: $
 *
 */

#ifndef PH_DATATYPES_H
#define PH_DATATYPES_H

#ifndef PH_PLATFORM_H_
#   include <ph_Platform.h>
#endif

/** \addtogroup phDatatypes
 *
 * \brief Contains definitions for portable data types used within this firmware.
 *
 * The data types defined here have to be used instead of standard C data types (e.g. int, char)
 * to allow easier portability of the firmware.
 * @{
 */

#ifndef PH_CONFIG_H
#   include <ph_Config.h>
#endif

#if !defined(__cplusplus) || defined(__arm__)
#ifndef __BOOL_DEFINED
#define __BOOL_DEFINED 1
/**
* \brief Boolean type
*/
#ifndef false
#define false    0
#endif
#ifndef true
#define true     (!false)
#endif
typedef uint32_t bool;
#endif
#endif

#ifndef TRUE
#define TRUE true
#endif

#ifndef FALSE
#define FALSE false
#endif

#ifndef NULL
#define NULL ((void*)0)  /**< Value used for NULL pointers */
#endif

#ifndef __float32_t_defined
typedef float float32_t;
#endif

/**
 * \brief Declaration of interrupt service routines.
 * This macro should be used for declaration of interrupt service routines. Example:
 * \code
PH_INTERRUPT(phSwp_Isr)
{
   ...
}
\endcode
 */
#define PH_INTERRUPT(ISR) void (ISR)(void)

#ifdef PHFL_ENABLE_STACK_CHECK
#   define PH_CALL_DIRECT_FUNCTION(fct) (fct)
#else
#   define PH_CALL_DIRECT_FUNCTION(fct)
#endif

#define STRINGIFY(a) #a

#if (PHFL_CONFIG_TARGET_PLATFORM == PH_CONFIG_TARGET_PLATFORM_DS5)
#    define PH_ALIGN(X)           __align(X)
#    define PH_NOINIT             __attribute__ ((section (".noinit")))
#    define PH_NOINIT_ALIGNED     __attribute__ ((aligned (4),section(".noinit")))
#    define PH_PACK_STRUCT_BEGIN  __packed
#    define PH_PACK_STRUCT_END
#    define PH_PLACE_FUNCTION_TO_SECTION(SECTIONNAME) \
                                  __attribute__ ((section (SECTIONNAME)))
#    define PH_UNUSED             __attribute__((unused))
#    define PH_USED               __attribute__((used))
#    define PH_WEAK               __attribute__((weak))
#elif (PHFL_CONFIG_TARGET_PLATFORM == PH_CONFIG_TARGET_PLATFORM_ARMGCC)
#    if (((__GNUC__ == 9) && (__GNUC_MINOR__ == 2)) || \
         ((__GNUC__ == 10) && ((__GNUC_MINOR__ == 2) || (__GNUC_MINOR__ == 3))) || \
         ((__GNUC__ == 12) && ((__GNUC_MINOR__ == 2) || (__GNUC_MINOR__ == 3))) || \
         ((__GNUC__ == 13) && (__GNUC_MINOR__ == 2)))
         /* Disable unaligned pointer value [-Waddress-of-packed-member] warnings */
#        define PH_DISABLE_PACKED_ALIGNED4_WARNING
#    endif /* end of GNUC version check */
#    define PH_ALIGN(a)           __attribute__ ((aligned (a)))
#    define PH_NOINIT             __attribute__ ((section (".noinit")))
#    define PH_NOINIT_ALIGNED     __attribute__ ((aligned (4),section(".noinit")))
#    define PH_PACK_STRUCT_BEGIN
#    define PH_PACK_STRUCT_END    __attribute__ (( packed ))
#    define PH_PLACE_FUNCTION_TO_SECTION(SECTIONNAME)                  \
                                  __attribute__ ((section (SECTIONNAME)))
#    define PH_UNUSED             __attribute__((unused))
#    define PH_USED               __attribute__((used))
#    define PH_WEAK               __attribute__((weak))
#elif (PHFL_CONFIG_TARGET_PLATFORM == PH_CONFIG_TARGET_PLATFORM_IAR)
#    define PH_ALIGN(X)           _Pragma(STRINGIFY(data_alignment = X))
#    define PH_NOINIT             __no_init
#    define PH_NOINIT_ALIGNED     _Pragma(STRINGIFY(data_alignment = 4)) __no_init
#    define PH_PLACE_FUNCTION_TO_SECTION(SECTION) \
                                  _Pragma(STRINGIFY(location = # SECTION))
#    define PH_PACK_STRUCT_BEGIN  __packed
#    define PH_PACK_STRUCT_END
#    define PH_UNUSED             /* Not used API parameters */
#    define PH_USED               /* TBU */
#    define PH_WEAK               __weak
#else /* PHFL_CONFIG_TARGET_PLATFORM */
#    error "Platform/configuration not supported"
#endif /* PHFL_CONFIG_TARGET_PLATFORM */

#if defined(__QMORE__) || defined(__DOXYGEN__)
#    undef PH_PLACE_FUNCTION_TO_SECTION
#    undef PH_PACK_STRUCT_BEGIN
#    undef PH_PACK_STRUCT_END
#    undef PH_UNUSED
#    undef PH_ALIGN
#    undef PH_NOINIT
#    undef PH_NOINIT_ALIGNED
#    undef __WFI

#    define PH_PLACE_FUNCTION_TO_SECTION(SECTIONNAME)
#    define PH_PACK_STRUCT_BEGIN
#    define PH_PACK_STRUCT_END
#    define PH_UNUSED
#    define PH_ALIGN(X)
#    define PH_NOINIT
#    define PH_NOINIT_ALIGNED
#    define __WFI()
#endif /*  __QMORE__ */

/**
 * \brief Pointer to a 32 bits register
 */
typedef volatile uint32_t * pReg32_t;

/* Satsfy the compiler regarding unused local variabled. */

#define UNUSED(PARAM) ((void)(PARAM))

/* Hint the compiler regarding unused parameters of the function.
 *
 * Usage,
 *
 *  void my_function(int UNUSED_PARAM(var))
 *
 */

#ifdef __UNUSED_PARAM
#   define UNUSED_PARAM UNUSED_PARAM
#else
#   define UNUSED_PARAM(PARAM)  PARAM __attribute__ ((__unused__))
#endif

/** @} */

#endif /* PH_DATATYPES_H */
