/*----------------------------------------------------------------------------*/
/* Copyright 2014-2015, 2022 NXP                                              */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** @file
 *
 * I2C Master event based mechanism implementation.
 *
 * Project:  PN7462AU
 *
 * $Date:
 * $Author:
 * $Revision:
 */
/* *****************************************************************************************************************
 * Includes
 * ***************************************************************************************************************** */
#include "ph_Datatypes.h"
#include "ph_NxpBuild.h"

#ifdef NXPBUILD__PHHAL_I2CM
#include "phhalI2CM.h"
#include "PN7462AU/PN7462AU_i2cm.h"
#include "phhalI2CM_Event.h"
#include "phOsal.h"

/* *****************************************************************************************************************
 * Internal Definitions
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Type Definitions
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Global and Static Variables
 * Total Size: NNNbytes
 * ***************************************************************************************************************** */
PH_NOINIT static uint32_t dwI2CM_TicksToWait;
PH_NOINIT static phOsal_EventObj_t      HwEventObj;                               /**< Event Object */
/* *****************************************************************************************************************
 * Private Functions Prototypes
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Public Functions
 * ***************************************************************************************************************** */

phStatus_t phhalI2CM_EventRegister(uint32_t dwTicksToWait PH_UNUSED)
{
    phStatus_t e_Status;
    dwI2CM_TicksToWait = dwTicksToWait;

    e_Status = phOsal_EventCreate( &(HwEventObj.EventHandle), &HwEventObj );

#if PHFL_PARAM_CHECK >= PH_PARAM_CHECK_SW_RESOURCE
    /* Was the event group created successfully */
    PH_RETURN_IF_TRUE_WITH_ERROR(( e_Status != PH_OSAL_SUCCESS ),RESOURCE_ERROR, I2CM)
#endif

    return PH_ERR_SUCCESS;
}

/**
 * phhalI2CM_EventHandler Event handler is called from Interrupt/Handler context.
 *
 */
phStatus_t phhalI2CM_EventHandler(uint8_t bId PH_UNUSED, uint32_t dwStatus PH_UNUSED)
{
#if PHFL_PARAM_CHECK >= PH_PARAM_CHECK_SW_RESOURCE
    phStatus_t e_Status;
#endif

#if PHFL_PARAM_CHECK >= PH_PARAM_CHECK_SW_RESOURCE
    e_Status = phOsal_EventPost( &(HwEventObj.EventHandle),E_OS_EVENT_OPT_NONE, (uint32_t)dwStatus, NULL );
    PH_RETURN_IF_TRUE_WITH_ERROR((e_Status != PH_OSAL_SUCCESS),RESOURCE_ERROR,I2CM)
#else
    phRtos_EventGroupSetBits( I2CM_TransactionEvent, dwStatus);
#endif

    return PH_ERR_SUCCESS;
}

phStatus_t phhalI2CM_EventWait(void)
{
    uint32_t dwBits = 0;

    phOsal_EventPend( &(HwEventObj.EventHandle), E_OS_EVENT_OPT_PEND_CLEAR_ON_EXIT, dwI2CM_TicksToWait,
            I2CMASTER_INT_STATUS_REG_I2C_BUS_ERROR_MASK
            | I2CMASTER_INT_STATUS_REG_NACK_MASK
            | I2CMASTER_INT_STATUS_REG_ARB_FAILURE_MASK
            | I2CMASTER_INT_STATUS_REG_TRN_COMPLETED_MASK
            | PH_ERR_ADDR_NAK_ERROR, &dwBits);

    if ( dwBits & (I2CMASTER_INT_STATUS_REG_I2C_BUS_ERROR_MASK
            | I2CMASTER_INT_STATUS_REG_NACK_MASK
            | I2CMASTER_INT_STATUS_REG_ARB_FAILURE_MASK
            | I2CMASTER_INT_STATUS_REG_TRN_COMPLETED_MASK
            | PH_ERR_ADDR_NAK_ERROR) )
    {
        return ((phStatus_t)dwBits);
    }
    else{
         return PH_ERR(OPERATION_TIMEDOUT,I2CM);
    }
}

void phhalI2CM_EventUnRegister(void)
{
    /* Delete an event group that was previously created by phhalI2CM_EventRegister. */
    phOsal_EventDelete( &(HwEventObj.EventHandle) );
}

/* *****************************************************************************************************************
 * Private Functions
 * ***************************************************************************************************************** */

#endif /* NXPBUILD__PHHAL_I2CM */
