/*----------------------------------------------------------------------------*/
/* Copyright 2017-2024 NXP                                                    */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** \file
* Generic ICode Application Component of Reader Library Framework.
* $Author: NXP $
* $Revision: $ (v07.11.00)
* $Date: $
*
*/

#ifndef PHALICODE_H
#define PHALICODE_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHAL_ICODE_SW

/** \defgroup phalICode_Sw Component : Software
 * @{
 */

#define PHAL_ICODE_SW_ID                                                0x01U   /**< ID for ICode Software layer. */

/** \brief Parameter structure for Icode Software layer implementation. */
typedef struct
{
    uint16_t wId;                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pPalSli15693DataParams;                                              /**< Pointer to the parameter structure of the underlying palSli15693 layer. */
    void * pCryptoDataParams;                                                   /**< Pointer to the parameter structure of the underlying Crypto layer for encryption / Decryption. */
    void * pCryptoRngDataParams;                                                /**< Pointer to the parameter structure of the underlying Crypto layer for random number generation. */
    void * pKeyStoreDataParams;                                                 /**< Pointer to the parameter structure of the underlying Keystore layer. */
    uint8_t aRnd_Challenge[10];                                                 /**< 10 byte buffer to store the random number generated by Challenge command. */
    uint8_t bBuffering;                                                         /**< Enable or disable the data bufferring. */
} phalICode_Sw_DataParams_t;

/**
 * \brief Initialization API for ICode software component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_Sw_Init(
        phalICode_Sw_DataParams_t * pDataParams,                                /**< [In] Pointer to this layers parameter structure. */
        uint16_t wSizeOfDataParams,                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pPalSli15693DataParams,                                          /**< [In] Pointer to the parameter structure of the underlying palSli15693 layer. */
        void * pCryptoDataParams,                                               /**< [In] Pointer to the parameter structure of the underlying Crypto layer for encryption / Decryption. */
        void * pCryptoRngDataParams,                                            /**< [In] Pointer to the parameter structure of the underlying Crypto layer for random number generation. */
        void * pKeyStoreDataParams                                              /**< [In] Pointer to the parameter structure of the underlying Keystore layer. */
);

/** end of phalICode_Sw
 * @}
 */
#endif /* NXPBUILD__PHAL_ICODE_SW */

#ifdef NXPBUILD__PHAL_ICODE

/** \defgroup phalICode ICode
 * \brief This component implements the commands of ICode products.
 * @{
 */

/** \name ICode Block Size. */
/* @{ */
#define PHAL_ICODE_BLOCK_SIZE                                               4U  /**< Macro to represent the ICode block size. */
/* @} */

/** \defgroup phalICode_Generic Generic
 * \brief This component implements the mandatory commands of ICode products.
 * @{
 */

/** \name Response error codes according to ISO15693 specification. */
/* @{ */
#define PHAL_ICODE_RESP_ERR_COMMAND_NOT_SUPPORTED                       0x01U   /**< The command is not supported, i.e. the request code is not recognized. */
#define PHAL_ICODE_RESP_ERR_COMMAND_NOT_RECOGNIZED                      0x02U   /**< The command is not recognized, for example: a format error occurred. */
#define PHAL_ICODE_RESP_ERR_COMMAND_OPTION_NOT_SUPPORTED                0x03U   /**< The command option is not supported. */
#define PHAL_ICODE_RESP_ERR_NO_INFORMATION                              0x0FU   /**< Error with no information given or a specific error code is not supported. */
#define PHAL_ICODE_RESP_ERR_BLOCK_NOT_AVAILABLE                         0x10U   /**< The specified block is not available (doesn't exist). */
#define PHAL_ICODE_RESP_ERR_BLOCK_LOCKED                                0x11U   /**< The specified block is already locked and thus cannot be locked again. */
#define PHAL_ICODE_RESP_ERR_CONTENT_CHANGE_FAILURE                      0x12U   /**< The specified block is locked and its content cannot be changed. */
#define PHAL_ICODE_RESP_ERR_BLOCK_PROGRAMMING_FAILURE                   0x13U   /**< The specified block was not successfully programmed. */
#define PHAL_ICODE_RESP_ERR_BLOCK_NOT_LOCKED                            0x14U   /**< The specified block was not successfully locked. */
#define PHAL_ICODE_RESP_ERR_BLOCK_PROTECTED                             0x15U   /**< The specified block is protected. */
#define PHAL_ICODE_RESP_ERR_GENERIC_CRYPTO_ERROR                        0x40U   /**< Generic cryptographic error. */
/* @} */

/** \name Custom error codes for ISO15693 response code. */
/* @{ */
#define PHAL_ICODE_ERR_CUSTOM_COMMANDS_ERROR        ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 0U)  /**< Custom commands Error codes. */
#define PHAL_ICODE_ERR_COMMAND_NOT_SUPPORTED        ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 1U)  /**< The command is not supported, i.e. the request code is not recognized. */
#define PHAL_ICODE_ERR_COMMAND_NOT_RECOGNIZED       ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 2U)  /**< The command is not recognized, for example: a format error occurred. */
#define PHAL_ICODE_ERR_COMMAND_OPTION_NOT_SUPPORTED ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 3U)  /**< The command option is not supported. */
#define PHAL_ICODE_ERR_NO_INFORMATION               ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 4U)  /**< Error with no information given or a specific error code is not supported. */
#define PHAL_ICODE_ERR_BLOCK_NOT_AVAILABLE          ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 5U)  /**< The specified block is not available (doesn't exist). */
#define PHAL_ICODE_ERR_BLOCK_LOCKED                 ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 6U)  /**< The specified block is already locked and thus cannot be locked again. */
#define PHAL_ICODE_ERR_CONTENT_CHANGE_FAILURE       ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 7U)  /**< The specified block is locked and its content cannot be changed. */
#define PHAL_ICODE_ERR_BLOCK_PROGRAMMING_FAILURE    ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 8U)  /**< The specified block was not successfully programmed. */
#define PHAL_ICODE_ERR_BLOCK_NOT_LOCKED             ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 9U)  /**< The specified block was not successfully locked. */
#define PHAL_ICODE_ERR_BLOCK_PROTECTED              ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 10U) /**< The specified block is protected. */
#define PHAL_ICODE_ERR_GENERIC_CRYPTO_ERROR         ((phStatus_t)PH_ERR_CUSTOM_BEGIN + 11U) /**< Generic cryptographic error. */
/* @} */

/** \name Request Flags according to ISO15693 specification. */
/* @{ */
#define PHAL_ICODE_FLAG_TWO_SUB_CARRIERS                                0x01U   /**< Sub-carrier_flag. If set two sub-carriers shall be used (VICC to VCD). Else a single sub-carrier
 *   frequency shall be used by the VICC.
 */
#define PHAL_ICODE_FLAG_DATA_RATE                                       0x02U   /**< Data_rate_flag. If set high data rate shall be used (VICC to VCD). Else low data rate shall
 *   be used.
 */
#define PHAL_ICODE_FLAG_INVENTORY                                       0x04U   /**< Inventory_flag. Shall be set for inventory request.
 *   If set flags 5 to 8 meaning is as per #PHAL_ICODE_FLAG_AFI, #PHAL_ICODE_FLAG_NBSLOTS and
 *   #PHAL_ICODE_FLAG_OPTION flags.
 *   If not set flags 5 to 8 meaning is as per #PHAL_ICODE_FLAG_SELECTED, #PHAL_ICODE_FLAG_ADDRESSED
 *   and #PHAL_ICODE_FLAG_OPTION flags.
 */
#define PHAL_ICODE_FLAG_PROTOCOL_EXTENSION                              0x08U   /**< Protocol_Extension_flag. If set protocol format is extended. Reserved for future use. No supported
 *   in current version.
 */
#define PHAL_ICODE_FLAG_SELECTED                                        0x10U   /**< Select_flag. If set only VICC in selected state shall respond. The Address_flag #PHAL_ICODE_FLAG_ADDRESSED
 *   shall be set to 0 and the UID field will not be included in the request. If not set Request shall be executed
 *   by any VICC according to the setting of Address_flag #PHAL_ICODE_FLAG_ADDRESSED.
 */
#define PHAL_ICODE_FLAG_AFI                                             0x10U   /**< AFI_flag. If set AFI field is present in request. If not set AFI field is not present in request. */
#define PHAL_ICODE_FLAG_ADDRESSED                                       0x20U   /**< Address_flag. If set request is addressed. UID field is included in request. It shall be executed only by the
 *   VICC whose UID matches the UID specified in the request. If not set request is not addressed. UID field is not
 *   included in request. It shall be executed by any VICC.
 */
#define PHAL_ICODE_FLAG_NBSLOTS                                         0x20U   /**< Nb_slots_flag. If set Use 1 slot instead of 16 slots for inventory. If not set use 16 slots. */
#define PHAL_ICODE_FLAG_OPTION                                          0x40U   /**< Option_flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise defined by
 *   the command.
 */
#define PHAL_ICODE_FLAG_FAST_DATA_RATE                                  0x80U   /**< RFU as per ISO15693 but this flag should be utilized for NTAG5 I2C product only. This flag is used to
 *   set the hardware RX Datarate to Fast_High (~53 kbps) for Fast_Low (~13 kbps). Also to note that this flag
 *   should not be used for any Inventory commands. Currently the support is for \ref phalICode_FastReadMultipleBlocks
 *   and \ref phalICode_ExtendedFastReadMultipleBlocks.
 */
/* @} */

/** \name Options */
/* @{ */
#define PHAL_ICODE_OPTION_OFF                           PH_OFF                  /**< Option Bit is cleared. */
#define PHAL_ICODE_OPTION_ON                            PH_ON                   /**< Option Bit is set. */
#define PHAL_ICODE_OPTION_DEFAULT                       PHAL_ICODE_OPTION_OFF   /**< Default Option. This flag represents #PHAL_ICODE_OPTION_OFF value. */
/*@}*/

/** \name AuthenticateMAM PurposeMAM2 options. */
/* @{ */
#define PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY_HF_RESET                0x08U   /**< Option to indicate PurposeMAM2 value as Privacy disable until HF reset. */
#define PHAL_ICODE_PURPOSE_MAM2_ENABLE_PRIVACY                          0x09U   /**< Option to indicate PurposeMAM2 value as Privacy enable. */
#define PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY                         0x0AU   /**< Option to indicate PurposeMAM2 value as Privacy disable. */
#define PHAL_ICODE_PURPOSE_MAM2_DESTROY                                 0x0BU   /**< Option to indicate PurposeMAM2 value as Destroy. */
/* @} */

/** \name Extended Get System Information parameter request fields. */
/* @{ */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_DEFAULT                          0x10U    /**< Option to indicate the default value of VICC response with memory addressing. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_DSFID                            0x01U    /**< Option to indicate the return of DSFID information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_AFI                              0x02U    /**< Option to indicate the return of AFI information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_VICC_MEM_SIZE                    0x04U    /**< Option to indicate the return of VICC memory size information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_IC_REFERENCE                     0x08U    /**< Option to indicate the return of IC Reference information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_MOI                              0x10U    /**< Option to indicate the return of MOI (memory addressing) information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_COMMAND_LIST                     0x20U    /**< Option to indicate the return of Supported Command List information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_CSI_INFORMATION                  0x40U    /**< Option to indicate the return of CSI information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_EXT_GET_SYS_INFO                 0x80U    /**< Option to indicate the return of Two byte length of Ext Get System information from VICC. */
/* @} */

/** \name ReadBuffer response verification options. */
/* @{ */
#define PHAL_ICODE_READ_BUFFER_VERIFY_OFF                               PH_OFF  /**< Option to return the received response from VICC without verification and decryption. */
#define PHAL_ICODE_READ_BUFFER_VERIFY_ON                                PH_ON   /**< Option to return the received response from VICC with verification and decryption. */
/* @} */

/** \name InventoryRead Extended options. */
/* @{ */
#define PHAL_ICODE_INVENTORY_READ_EXT_DEFAULT                           0x00U    /**< Default option for InventoryReadExtended command. */
#define PHAL_ICODE_INVENTORY_READ_EXT_EAS_MODE                          0x01U    /**< Option to indicate that the VICC with EAS enabled will respond. */
#define PHAL_ICODE_INVENTORY_READ_EXT_UID_MODE                          0x02U    /**< Option to indicate complete UID transmission. */
#define PHAL_ICODE_INVENTORY_READ_EXT_CID_COMPARE                       0x04U    /**< Option to indicate transmission of 16 bit CID and only tags with the same CID will respond. */
#define PHAL_ICODE_INVENTORY_READ_EXT_CID_RESPONSE                      0x08U    /**< Option to indicate transmission of Custom ID. */
#define PHAL_ICODE_INVENTORY_READ_EXT_SKIP_DATA                         0x10U    /**< Option to indicate No user memory data are requested from the tag, first block number byte and
 *   number of blocks byte shall not be transmitted in the command. */
#define PHAL_ICODE_INVENTORY_READ_EXT_QUIET                             0x20U    /**< Option to indicate the state change to Quiet after response. */
#define PHAL_ICODE_INVENTORY_READ_EXT_PERSIST_QUIET                     0x40U    /**< Option to indicate the state change to Persistent Quiet after response. */
#define PHAL_ICODE_INVENTORY_READ_EXT_PERSIST_QUIET_RESPONSE            0x60U    /**< Option to indicate that tags in the PERSISTENT QUIET state will respond to the command. */
/* @} */

/** \name SetPassword Password Identifier options. */
/* @{ */
#define PHAL_ICODE_SET_PASSWORD_READ                                    0x01U    /**< Option to indicate Password Identifier as Read. */
#define PHAL_ICODE_SET_PASSWORD_WRITE                                   0x02U    /**< Option to indicate Password Identifier as Write. */
#define PHAL_ICODE_SET_PASSWORD_PRIVACY                                 0x04U    /**< Option to indicate Password Identifier as Privacy. */
#define PHAL_ICODE_SET_PASSWORD_DESTROY                                 0x08U    /**< Option to indicate Password Identifier as Destroy. */
#define PHAL_ICODE_SET_PASSWORD_EAS                                     0x10U    /**< Option to indicate Password Identifier as EAS. */
/* @} */

/** \name ProtectPage Protection Status options for the ICode products which do not describe the pages as High and Low. */
/* @{ */

#define PHAL_ICODE_PROTECT_PAGE_PUBLIC                                  0x00U   /**< Option to indicate protection status as public. */
#define PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD                0x01U   /**< Option to indicate protection status as
                                                                                 *      \arg 32-bit Password Protection: Read and Write protection by Read password.
                                                                                 *      \arg 64-bit Password Protection: Read and Write protection by Read plus Write password.
                                                                                 *      \arg ICode DNA or other products that supports AES authentication, Mutul authentication
                                                                                 *           with a key with read privilege is required.
                                                                                 */
#define PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD                          0x10U   /**< Option to indicate protection status as
                                                                                 *      \arg 32-bit Password Protection: Write protection by Write password.
                                                                                 *      \arg 64-bit Password Protection: Write protection by Read plus Write password.
                                                                                 *      \arg ICode DNA or other products that supports AES authentication, Mutul authentication
                                                                                 *           with a key with write privilege is required.
                                                                                 */
#define PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE            0x11U   /**< Option to indicate protection status as
                                                                                 *      \arg 32-bit Password Protection: Read protected by Read and Write protection by Write password.
                                                                                 *      \arg 64-bit Password Protection: Read and Write protection by Read plus Write password.
                                                                                 *      \arg ICode DNA or other products that supports AES authentication, Mutul authentication
                                                                                 *           with a key with read and write privileges is required.
                                                                                 */
/* @} */

/** \name Extended Protection Status options for the ICode products which describes the pages as High and Low. */
/* @{ */
#define PHAL_ICODE_PROTECT_PAGE_READ_LOW                                0x01U   /**< Option to indicate Read protection for Lower pages. */
#define PHAL_ICODE_PROTECT_PAGE_WRITE_LOW                               0x02U   /**< Option to indicate Write protection for Lower pages. */
#define PHAL_ICODE_PROTECT_PAGE_READ_HIGH                               0x10U   /**< Option to indicate Read protection for Higher pages. */
#define PHAL_ICODE_PROTECT_PAGE_WRITE_HIGH                              0x20U   /**< Option to indicate Write protection for Higher pages. */

/* @} */

/** \name ParameterRequest and ParameterSelect Bitrate options. */
/* @{ */
#define PHAL_ICODE_PARAMETERS_BITRATE_26KBPS_BOTH_DIRECTIONS            0x00U    /**< Option to indicate the 26kbps bitrate support in both directions. */
#define PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC                   0x01U    /**< Option to indicate the 53kbps bitrate support from VCD to VICC. */
#define PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC                  0x02U    /**< Option to indicate the 106kbps bitrate support from VCD to VICC. */
#define PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC                  0x04U    /**< Option to indicate the 212kbps bitrate support from VCD to VICC. */
#define PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD                   0x10U    /**< Option to indicate the 53kbps bitrate support from VICC to VCD. */
#define PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD                  0x20U    /**< Option to indicate the 106kbps bitrate support from VICC to VCD. */
#define PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD                  0x40U    /**< Option to indicate the 212kbps bitrate support from VICC to VCD. */
/* @} */

/** \name ParameterRequest and ParameterSelect Timing options. */
/* @{ */
#define PHAL_ICODE_PARAMETERS_TIMING_320_9_US                           0x00U    /**< Option to indicate that VICC supports 320.9us timing only. */
#define PHAL_ICODE_PARAMETERS_TIMING_160_5_US                           0x01U    /**< Option to indicate that VICC supports 160.5us timing only. */
#define PHAL_ICODE_PARAMETERS_TIMING_80_2_US                            0x02U    /**< Option to indicate that VICC supports 80.2us timing only. */
#define PHAL_ICODE_PARAMETERS_TIMING_SAME_BOTH_DIRECTIONS               0x10U    /**< Option to indicate that same timing on both the direction is compulsory. */

/* @} */

/** \name GetConfig / SetConfig options. */
/* @{ */
#define PHAL_ICODE_CONFIG_FLAGS                                         0x00U    /**< Option to Get / Set the Flags information. */
#define PHAL_ICODE_CONFIG_ADD_INFO                                      0x01U    /**< Option to Get the additional information. */
#define PHAL_ICODE_CONFIG_TIMEOUT_US                                    0x02U    /**< Option to Get the timeout value in milliseconds. */
#define PHAL_ICODE_CONFIG_TIMEOUT_MS                                    0x03U    /**< Option to Get the timeout value in microseconds. */
#define PHAL_ICODE_CONFIG_ENABLE_BUFFERING                              0x04U    /**< Option to Get / Set the buffering of response data to be enabled or disabled.By default this flag will be disabled. */
/* @} */

/** \name GetConfig / SetConfig value for #PHAL_ICODE_CONFIG_ENABLE_BUFFERING configuration. */
/* @{ */
#define PHAL_ICODE_DISABLE                                              0x00U    /**< Option to disable the buffering of response data. */
#define PHAL_ICODE_ENABLE                                               0x01U    /**< Option to enable the buffering of response data. */
/* @} */

#define PHAL_ICODE_TIMEOUT_SHORT_US                                     324U    /**< Timeout T1 in us (ISO/IEC15693, 9.1.4.). */
#define PHAL_ICODE_TIMEOUT_LONG_US                                      20000U  /**< Extended timeout for some commands. */

/** \name Options for type of tags. */
/* @{ */
#define PHAL_ICODE_TAG_TYPE_UNKNOWN                                     0xFFFFU  /**< Option to Get the tag type as Unknown. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI                                   0x0001U  /**< Option to Get the tag type as ICode SLI. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_S                                 0x0002U  /**< Option to Get the tag type as ICode SLI-S. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_L                                 0x0003U  /**< Option to Get the tag type as ICode SLI-L. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLIX                                  0x5001U  /**< Option to Get the tag type as ICode SLIX. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLIX_S                                0x5002U  /**< Option to Get the tag type as ICode SLIX-S. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLIX_L                                0x5003U  /**< Option to Get the tag type as ICode SLIX-L. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_X2                                0x0801U  /**< Option to Get the tag type as ICode SLIX2. */
#define PHAL_ICODE_TAG_TYPE_ICODE_DNA                                   0x1801U  /**< Option to Get the tag type as ICode DNA. */
#define PHAL_ICODE_TAG_TYPE_ICODE_NTAG5_I2C                             0x5801U  /**< Option to Get the tag type as ICode NTAG5 I2C. */
/* @} */

#ifdef  NXPRDLIB_REM_GEN_INTFS
#include "../comps/phalICode/src/Sw/phalICode_Sw.h"

#define phalICode_ReadSingleBlock(pDataParams, bOption, bBlockNo, ppData, pDataLen)  \
        phalICode_Sw_ReadSingleBlock((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo, ppData, pDataLen)

#define phalICode_WriteSingleBlock(pDataParams, bOption, bBlockNo, pData, bDataLen)  \
        phalICode_Sw_WriteSingleBlock((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo, pData, bDataLen)

#define phalICode_LockBlock(pDataParams, bOption, bBlockNo)  \
        phalICode_Sw_LockBlock((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo)

#define phalICode_ReadMultipleBlocks(pDataParams, bOption, bBlockNo, bNumBlocks, pData,  pDataLen)  \
        phalICode_Sw_ReadMultipleBlocks((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo, bNumBlocks, pData,  pDataLen)

#define phalICode_WriteAFI(pDataParams, bOption, bAfi)  \
        phalICode_Sw_WriteAFI((phalICode_Sw_DataParams_t *)pDataParams, bOption, bAfi)

#define phalICode_LockAFI(pDataParams, bOption)  \
        phalICode_Sw_LockAFI((phalICode_Sw_DataParams_t *)pDataParams, bOption)

#define phalICode_WriteDSFID(pDataParams, bOption, bDsfid)  \
        phalICode_Sw_WriteDSFID((phalICode_Sw_DataParams_t *)pDataParams, bOption, bDsfid)

#define phalICode_LockDSFID(pDataParams, bOption)  \
        phalICode_Sw_LockDSFID((phalICode_Sw_DataParams_t *)pDataParams, bOption)

#define phalICode_GetSystemInformation(pDataParams, ppSystemInfo, pSystemInfoLen)  \
        phalICode_Sw_GetSystemInformation((phalICode_Sw_DataParams_t *)pDataParams, ppSystemInfo, pSystemInfoLen)

#define phalICode_GetMultipleBlockSecurityStatus(pDataParams, bBlockNo, bNoOfBlocks, pStatus, pStatusLen)  \
        phalICode_Sw_GetMultipleBlockSecurityStatus((phalICode_Sw_DataParams_t *)pDataParams, bBlockNo, bNoOfBlocks, pStatus, pStatusLen)

#define phalICode_FastReadMultipleBlocks(pDataParams, bOption, bBlockNo, bNumBlocks, pData, pDataLen)  \
        phalICode_Sw_FastReadMultipleBlocks((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo, bNumBlocks, pData, pDataLen)

#define phalICode_ExtendedReadSingleBlock(pDataParams, bOption, wBlockNo, ppData, pDataLen)  \
        phalICode_Sw_ExtendedReadSingleBlock((phalICode_Sw_DataParams_t *)pDataParams, bOption, wBlockNo, ppData, pDataLen)

#define phalICode_ExtendedWriteSingleBlock(pDataParams, bOption, wBlockNo, pData, bDataLen)  \
        phalICode_Sw_ExtendedWriteSingleBlock((phalICode_Sw_DataParams_t *)pDataParams, bOption, wBlockNo, pData, bDataLen)

#define phalICode_ExtendedLockBlock(pDataParams, bOption, wBlockNo)   \
        phalICode_Sw_ExtendedLockBlock((phalICode_Sw_DataParams_t *)pDataParams, bOption, wBlockNo)

#define phalICode_ExtendedReadMultipleBlocks(pDataParams, bOption, wBlockNo, wNumBlocks, pData, pDataLen)  \
        phalICode_Sw_ExtendedReadMultipleBlocks((phalICode_Sw_DataParams_t *)pDataParams, bOption, wBlockNo, wNumBlocks, pData, pDataLen)

#ifdef NXPBUILD__PH_CRYPTOSYM
#define  phalICode_AuthenticateTAM1(pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard, pDivInput, bDivLen)  \
        phalICode_Sw_AuthenticateTAM1((phalICode_Sw_DataParams_t *)pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard)

#define phalICode_AuthenticateMAM(pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard, bPurposeMAM2, pDivInput, bDivLen) \
        phalICode_Sw_AuthenticateMAM((phalICode_Sw_DataParams_t *)pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard, bPurposeMAM2)

#define phalICode_Challenge(pDataParams, bKeyNoCard)  \
        phalICode_Sw_Challenge((phalICode_Sw_DataParams_t *)pDataParams, bKeyNoCard)

#define phalICode_ReadBuffer(pDataParams, bVerify, bKeyNo, bKeyVer, ppResponse, pRespLen)  \
        phalICode_Sw_ReadBuffer((phalICode_Sw_DataParams_t *)pDataParams, bVerify, bKeyNo, bKeyVer, ppResponse, pRespLen)
#endif /* NXPBUILD__PH_CRYPTOSYM */

#define phalICode_ExtendedGetSystemInformation(pDataParams, bInfoParams, ppSystemInfo, pSystemInfoLen) \
        phalICode_Sw_ExtendedGetSystemInformation((phalICode_Sw_DataParams_t *)pDataParams, bInfoParams, ppSystemInfo, pSystemInfoLen)

#define phalICode_ExtendedGetMultipleBlockSecurityStatus(pDataParams, wBlockNo, wNoOfBlocks, pStatus, pStatusLen) \
        phalICode_Sw_ExtendedGetMultipleBlockSecurityStatus((phalICode_Sw_DataParams_t *)pDataParams, wBlockNo, wNoOfBlocks, pStatus, pStatusLen)

#define phalICode_ExtendedFastReadMultipleBlocks(pDataParams, bOption, wBlockNo, wNumBlocks, pData, pDataLen) \
        phalICode_Sw_ExtendedFastReadMultipleBlocks((phalICode_Sw_DataParams_t *)pDataParams, bOption, wBlockNo, wNumBlocks, pData, pDataLen)

#define phalICode_SetEAS(pDataParams, bOption)  \
        phalICode_Sw_SetEAS((phalICode_Sw_DataParams_t *)pDataParams, bOption)

#define phalICode_ResetEAS(pDataParams, bOption) \
        phalICode_Sw_ResetEAS((phalICode_Sw_DataParams_t *)pDataParams, bOption)

#define phalICode_LockEAS(pDataParams, bOption) \
        phalICode_Sw_LockEAS((phalICode_Sw_DataParams_t *)pDataParams, bOption)

#define phalICode_EASAlarm(pDataParams, bOption, pEasIdValue, bEasIdMaskLen, ppEas, pEasLen) \
        phalICode_Sw_EASAlarm((phalICode_Sw_DataParams_t *)pDataParams, bOption, pEasIdValue, bEasIdMaskLen, ppEas, pEasLen)

#define phalICode_PasswordProtectEAS(pDataParams) \
        phalICode_Sw_PasswordProtectEAS((phalICode_Sw_DataParams_t *)pDataParams)

#define phalICode_PasswordProtectAFI(pDataParams) \
        phalICode_Sw_PasswordProtectAFI((phalICode_Sw_DataParams_t *)pDataParams)

#define phalICode_WriteEASID(pDataParams, bOption, pEasIdValue) \
        phalICode_Sw_WriteEAS_ID((phalICode_Sw_DataParams_t *)pDataParams, bOption, pEasIdValue)

#define phalICode_ReadEPC(pDataParams, ppEpc, pEpcLen) \
        phalICode_Sw_ReadEPC((phalICode_Sw_DataParams_t *)pDataParams, ppEpc, pEpcLen)

#define phalICode_GetNXPSystemInformation(pDataParams, ppSystemInfo, pSystemInfoLen) \
        phalICode_Sw_GetNXPSystemInformation((phalICode_Sw_DataParams_t *)pDataParams, ppSystemInfo, pSystemInfoLen)

#define phalICode_GetRandomNumber(pDataParams, ppRnd, pRndLen) \
        phalICode_Sw_GetRandomNumber((phalICode_Sw_DataParams_t *)pDataParams, ppRnd, pRndLen)

#define phalICode_SetPassword(pDataParams, bOption, bPwdIdentifier, pXorPwd) \
        phalICode_Sw_SetPassword((phalICode_Sw_DataParams_t *)pDataParams, bOption, bPwdIdentifier, pXorPwd)

#define phalICode_WritePassword(pDataParams, bOption, bPwdIdentifier, pPwd) \
        phalICode_Sw_WritePassword((phalICode_Sw_DataParams_t *)pDataParams, bOption, bPwdIdentifier, pPwd)

#define phalICode_LockPassword(pDataParams, bOption, bPwdIdentifier) \
        phalICode_Sw_LockPassword((phalICode_Sw_DataParams_t *)pDataParams, bOption, bPwdIdentifier)

#define phalICode_ProtectPage(pDataParams, bOption, bPPAdd_PageNo, bProtectionStatus) \
        phalICode_Sw_ProtectPage((phalICode_Sw_DataParams_t *)pDataParams, bOption, bPPAdd_PageNo, bProtectionStatus)

#define phalICode_LockPageProtectionCondition(pDataParams, bOption, bPageNo) \
        phalICode_Sw_LockPageProtectionCondition((phalICode_Sw_DataParams_t *)pDataParams, bOption, bPageNo)

#define phalICode_GetMultipleBlockProtectionStatus(pDataParams, bBlockNo, bNoOfBlocks, pProtectionStates, pNumReceivedStates) \
        phalICode_Sw_GetMultipleBlockProtectionStatus((phalICode_Sw_DataParams_t *)pDataParams, bBlockNo, bNoOfBlocks, pProtectionStates, pNumReceivedStates)

#define phalICode_Destroy(pDataParams, bOption, pXorPwd) \
        phalICode_Sw_Destroy((phalICode_Sw_DataParams_t *)pDataParams, bOption, pXorPwd)

#define  phalICode_EnablePrivacy(pDataParams, bOption, pXorPwd) \
        phalICode_Sw_EnablePrivacy((phalICode_Sw_DataParams_t *)pDataParams, bOption, pXorPwd)

#define phalICode_64BitPasswordProtection(pDataParams) \
        phalICode_Sw_64BitPasswordProtection((phalICode_Sw_DataParams_t *)pDataParams)

#define phalICode_ReadSignature(pDataParams, ppSign, pSignLen) \
        phalICode_Sw_ReadSignature((phalICode_Sw_DataParams_t *)pDataParams, ppSign, pSignLen)

#define phalICode_ReadConfig(pDataParams, bOption, bBlockAddr, bNoOfBlocks, ppData, pDataLen) \
        phalICode_Sw_ReadConfig((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockAddr, bNoOfBlocks, ppData, pDataLen)

#define phalICode_WriteConfig(pDataParams, bOption, bBlockAddr, pData) \
        phalICode_Sw_WriteConfig((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockAddr, pData)

#define phalICode_PickRandomID(pDataParams) \
        phalICode_Sw_PickRandomID((phalICode_Sw_DataParams_t *)pDataParams)

#define phalICode_ReadTT(pDataParams, bOption, ppResponse, pRespLen) \
        phalICode_Sw_ReadTT((phalICode_Sw_DataParams_t *)pDataParams, bOption, ppResponse, pRespLen)

#define phalICode_ParameterRequest(pDataParams, pBitRate, pTiming) \
        phalICode_Sw_ParameterRequest((phalICode_Sw_DataParams_t *)pDataParams, pBitRate, pTiming)

#define  phalICode_ParameterSelect(pDataParams, bBitRate, bTiming) \
        phalICode_Sw_ParameterSelect((phalICode_Sw_DataParams_t *)pDataParams, bBitRate, bTiming)

#define phalICode_ReadSRAM(pDataParams, bOption, bBlockNo, bNumBlocks, pData, pDataLen) \
        phalICode_Sw_ReadSRAM((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo, bNumBlocks, pData, pDataLen)

#define phalICode_WriteSRAM(pDataParams, bOption, bBlockNo, bNumBlocks, pData, wDataLen) \
        phalICode_Sw_WriteSRAM((phalICode_Sw_DataParams_t *)pDataParams, bOption, bBlockNo, bNumBlocks, pData, wDataLen)

#define phalICode_I2CMRead(pDataParams, bI2CParam, bDataLen, pData) \
        phalICode_Sw_I2CMRead((phalICode_Sw_DataParams_t *)pDataParams, bI2CParam, bDataLen, pData)

#define phalICode_I2CMWrite(pDataParams, bI2CParam, pData, bDataLen) \
        phalICode_Sw_I2CMWrite((phalICode_Sw_DataParams_t *)pDataParams, bI2CParam, pData, bDataLen)

#define phalICode_GetConfig(pDataParams, wConfig, pValue) \
        phalICode_Sw_GetConfig((phalICode_Sw_DataParams_t *)pDataParams, wConfig, pValue)

#define  phalICode_SetConfig(pDataParams, wConfig, wValue) \
        phalICode_Sw_SetConfig((phalICode_Sw_DataParams_t *)pDataParams, wConfig, wValue)

#define phalICode_GetTagType(pDataParams, pTagType) \
        phalICode_Sw_GetTagType((phalICode_Sw_DataParams_t *)pDataParams, pTagType)

#else

/** \defgroup phalICode_Optional Commands_Optional
 * \brief This component implements the optional commands of ICode products.
 * @{
 */

/**
 * \brief Performs a Single block read command. When receiving the Read Single Block command, the VICC shall read the requested block and send
 * back its value in the response. If the Option_flag (bOption = #PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall return the block
 * security status, followed by the block value. If it is not set (bOption = #PHAL_ICODE_OPTION_OFF), the VICC shall return only the block value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, block Security Status information is not available. Only block data
         *           is available. Format will be 4 byte data
         *      \arg If #PHAL_ICODE_OPTION_ON, both block Security Status information and Block Data is available.
         *           Format of the response will be Status, 4 byte data
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t ** ppData,                                                      /**< [Out] Information received from VICC in with respect to bOption parameter information. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a Single block write command. When receiving the Write single block command, the VICC shall write the requested block with the
 * data contained in the request and report the success of the operation in the response. If the Option_flag (bOption = PHAL_ICODE_OPTION_ON)
 * is set in the request, the VICC shall wait for the reception of an EOF from the VCD and upon such reception shall return its response.
 * If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return its response when it has completed the write operation starting
 * after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance of  32/fc and latest after 20 ms upon
 * detection of the rising edge of the EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WriteSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the write operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bBlockNo,                                                       /**< [In] Block number to which the data should be written. */
        uint8_t * pData,                                                        /**< [In] Information to be written to the specified block number. */
        uint8_t bDataLen                                                        /**< [In] Number of bytes to be written. */
);

/**
 * \brief Performs a Lock block command. When receiving the Lock block command, the VICC shall lock permanently the requested block. If the
 * Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall wait for the reception of an EOF from the VCD
 * and upon such reception shall return its response. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return its
 * response when it has completed the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc
 * (302 us) with a total tolerance of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_LockBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the lock operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bBlockNo                                                        /**< [In] Block number which should be locked. */
);

/**
 * \brief Performs a Multiple block read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s) and send
 * back its value in the response. If the Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall return the block
 * security status, followed by the block value sequentially block by block. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall
 * return only the block value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
         *           is available. Format will be 4 byte data 1, 4 byte data 2, 4 byte data N
         *      \arg If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
         *           Format of the response will be Status 1, 4 byte data 1, Status 2, 4 byte data 2, Status N,
         *           4 byte data N
         *           Where 1, 2 N is the block number.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC in with respect to bOption parameter information. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a WriteAFI command. When receiving the Write AFI request, the VICC shall write the AFI value into its memory.
 * If the  Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall wait for the reception of an EOF
 * from the VCD and upon such reception shall return its response. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC
 * shall return its response when it has completed the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a
 * multiple of 4096/fc (302 us) with a total tolerance of  32/fc and latest after 20 ms upon detection of the rising edge of the
 * EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WriteAFI(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the write operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bAfi                                                            /**< [In] Value of Application Family Identifier. */
);

/**
 * \brief Performs a LockAFI command. When receiving the Lock AFI request, the VICC shall lock the AFI value permanently into its memory.
 * If the  Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall wait for the reception of an EOF from the
 * VCD and upon such reception shall return its response. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return its
 * response when it has completed the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us)
 * with a total tolerance of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_LockAFI(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the lock operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
);

/**
 * \brief Performs WriteDSFID command. When receiving the Write DSFID request, the VICC shall write the DSFID value into its memory.
 * If the  Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall wait for the reception of an EOF
 * from the VCD and upon such reception shall return its response. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC
 * shall return its response when it has completed the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a
 * multiple of 4096/fc (302 us) with a total tolerance of  32/fc and latest after 20 ms upon detection of the rising edge of the
 * EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WriteDSFID(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the write operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bDsfid                                                          /**< [In] Value of DSFID (data storage format identifier). */
);

/**
 * \brief Performs LockDSFID command. When receiving the Lock DSFID request, the VICC shall lock the DSFID value permanently into its memory.
 * If the  Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall wait for the reception of an EOF from the
 * VCD and upon such reception shall return its response. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return its
 * response when it has completed the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us)
 * with a total tolerance of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_LockDSFID(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the lock operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
);

/**
 * \brief Performs GetSystemInformation command. This command allows for retrieving the system information value from the VICC.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_GetSystemInformation(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppSystemInfo,                                                /**< [Out] The system information of the VICC. */
        uint16_t * pSystemInfoLen                                               /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs GetMultipleBlockSecurityStatus. When receiving the Get multiple block security status command, the VICC
 * shall send back the block security status.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the Status buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_GetMultipleBlockSecurityStatus(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bBlockNo,                                                       /**< [In] Block number for which the status should be returned. */
        uint8_t bNoOfBlocks,                                                    /**< [In] Number of blocks to be used for returning the status. */
        uint8_t * pStatus,                                                      /**< [Out] The status of the block number mentioned in bBlockNo until bNumBlocks. */
        uint16_t * pStatusLen                                                   /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a Multiple block fast read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s)
 * and send back its value in the response. If the Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall return the
 * block security status, followed by the block value sequentially block by block. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC
 * shall return only the block value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_FastReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
         *           is available. Format will be 4 byte data 1, 4 byte data 2, 4 byte data N
         *      \arg If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
         *           Format of the response will be Status 1, 4 byte data 1, Status 2, 4 byte data 2,  Status N,
         *           4 byte data N
         *           Where 1, 2  N is the block number.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC in with respect to bOption parameter information. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a Extended Single block read command. When receiving the Extended Read Single Block command, the VICC shall read the
 * requested block and send back its value in the response. If a VICC supports Extended read single block command, it shall also support
 * Read single block command for the first 256 blocks of memory. If the Option_flag (bOption = #PHAL_ICODE_OPTION_ON) is set in the request,
 * the VICC shall return the block security status, followed by the block value. If it is not set (bOption = #PHAL_ICODE_OPTION_OFF), the
 * VICC shall return only the block value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedReadSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, block Security Status information is not available. Only block data
         *           is available. Format will be 4 byte data
         *      \arg If #PHAL_ICODE_OPTION_ON, both block Security Status information and Block Data is available.
         *           Format of the response will be Status, 4 byte data
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint16_t wBlockNo,                                                      /**< [In] Block number from where the data to be read. */
        uint8_t ** ppData,                                                      /**< [Out] Information received from VICC in with respect to bOption parameter information. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a Extended Single block Write command. When receiving the Extended write single block command, the VICC shall write the
 * requested block with the data contained in the request and report the success of the operation in the response. If a VICC supports
 * Extended write single block command, it shall also support Write single block command for the first 256 blocks of memory.
 *
 * If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return its response when it has completed the write operation starting
 * after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance of  32/fc and latest after 20 ms upon
 * detection of the rising edge of the EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedWriteSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the write operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint16_t wBlockNo,                                                      /**< [In] Block number to which the data should be written. */
        uint8_t * pData,                                                        /**< [In] Information to be written to the specified block number. */
        uint8_t bDataLen                                                        /**< [In] Number of bytes to be written. */
);

/**
 * \brief Performs a Extended Lock block command. When receiving the Lock block command, the VICC shall lock permanently the requested
 * block. If a VICC supports Extended lock block command, it shall also support Lock block command for the first 256 blocks of memory.
 * If the Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall wait for the reception of an EOF from the
 * VCD and upon such reception shall return its response. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return its
 * response when it has completed the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc
 * (302 us) with a total tolerance of 32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedLockBlock (
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the lock operation
         *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
         *           of  32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
         *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
         *           shall return its response.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint16_t wBlockNo                                                       /**< [In] Block number which should be locked. */
);

/**
 * \brief Performs a Extended Multiple block read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s)
 * and send back its value in the response. If a VICC supports Extended read multiple blocks command, it shall also support Read multiple blocks
 * command for the first 256 blocks of memory.
 *
 * If the Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall return the block security status, followed by the block
 * value sequentially block by block. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return only the block value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
         *           is available. Format will be 4 byte data 1, 4 byte data 2,  4 byte data N
         *      \arg If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
         *           Format of the response will be Status 1, 4 byte data 1, Status 2, 4 byte data 2,  Status N,
         *           4 byte data N
         *           Where 1, 2  N is the block number.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint16_t wBlockNo,                                                      /**< [In] Block number from where the data to be read. */
        uint16_t wNumBlocks,                                                    /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC in with respect to bOption parameter information. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes.*/
);

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Performs TAM1 authentication with the card.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_AuthenticateTAM1(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware keystore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyNoCard,                                                     /**< [In] Block number of the AES key available in the card.*/
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only*/
        uint8_t bDivLen                                                         /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
);

/**
 * \brief Performs TAM-2 authentication with the card.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_AuthenticateTAM2(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware keystore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyNoCard,                                                     /**< [In] Block number of the AES key available in the card.*/
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only. */
        uint8_t bDivLen,                                                        /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
        uint8_t bBlockSize,                                                     /**< [In] To select the size of custom data block to be used.
                                                                                          The value should either be 0x00 for 16 bit block size or 0x01 for 64 bit block size. As per ISO 29167 */
        uint8_t bBlockCount,                                                    /**< [In] To select the custom data block to be used from the offset specified.
                                                                                          The BlockCount range is from 1 - 16. */
        uint8_t bProfile,                                                       /**< [In] To select one of the memory profiles supported by the tag.
                                                                                          The Profile range is from 0 - 15. As per ISO 29167  */
        uint8_t bProtMode,                                                      /**< [In] To specify the mode of operation to be used for encryption/decryption.
                                                                                          The ProtMode ranges form 0 - 3. As per ISO 29167 */
        uint16_t wOffset,                                                       /**< [In] To set the offset for the specified profile. The Offset ranges form 0 - 4095. As per ISO 29167 */
        uint8_t * pCustomData,                                                  /**< [Out] The custom data returned by the card. */
        uint16_t * pCustomDataLen                                               /**< [Out] The length of custom data returned. */
);

/**
 * \brief Performs MAM authentication with the card. Both the MAM part 1 and MAM part 2 authentication are
 * carried out internally by this interface.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_AuthenticateMAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware keystore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyNoCard,                                                     /**< [In] Block number of the AES key available in the card.*/
        uint8_t bPurposeMAM2,                                                   /**< [In] The purpose MAM2 data to to be used.
         *          \arg #PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY_HF_RESET
         *          \arg #PHAL_ICODE_PURPOSE_MAM2_ENABLE_PRIVACY
         *          \arg #PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY
         *          \arg #PHAL_ICODE_PURPOSE_MAM2_DESTROY
         */
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only*/
        uint8_t bDivLen                                                         /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
);

/**
 * \brief Performs tag authentication with the card. This is another method of authenticating with the card.
 * Here the TAM1 challenge message is sent to the card. The card does not respond for this command. To verify
 * if this command was success the command \ref phalICode_ReadBuffer should be called.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_Challenge(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNoCard                                                      /**< [In] Block number of the AES key available in the card.*/
);

/**
 * \brief Reads the crypto calculation result of previous Challenge command. If the Challenge Command was success,
 * Then the encrypted response will be returned. The response will be same as TAM1 response format. If verification
 * is enabled (i.e. bVerify = 0x01), The encrypted response will be decrypted and the random number generated by the
 * Challenge command will be compared against the received one. If fails AUTH_ERROR will be returned.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadBuffer(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bVerify,                                                        /**< [In] To verify the received data with the random number generated by Challenge command.
         *        \arg #PHAL_ICODE_READ_BUFFER_VERIFY_OFF
         *        \arg #PHAL_ICODE_READ_BUFFER_VERIFY_ON
         */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t ** ppResponse,                                                  /**< [Out] If verification is enabled the decrypted response data will be available. Also
         *         the response will be verified with the random number generated by
         *         \ref phalICode_Challenge command.
         *         If verification is disabled the encrypted response data will be available.
         */
        uint16_t * pRespLen                                                     /**< [Out] Length of available bytes in ppResponse buffer. */
);

#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Performs ExtendedGetSystemInformation command. This command allows for retrieving the system information value
 * from the VICC and shall be supported by the VICC if extended memory or security functionalities are supported by the VICC.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedGetSystemInformation(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bInfoParams,                                                    /**< [In] Extend Get System Information parameter request fields.
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_DEFAULT
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_DSFID
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_AFI
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_VICC_MEM_SIZE
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_IC_REFERENCE
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_MOI
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_COMMAND_LIST
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_CSI_INFORMATION
         *        \arg #PHAL_ICODE_INFO_PARAMS_REQUEST_EXT_GET_SYS_INFO
         */
        uint8_t ** ppSystemInfo,                                                /**< [Out] The system information of the VICC. */
        uint16_t * pSystemInfoLen                                               /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs ExtendedGetMultipleBlockSecurityStatus. When receiving the Extended Get multiple block security status
 * command, the VICC shall send back the block security status. The blocks are numbered from 0000 to FFFF (0 - 65535).
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the Status buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedGetMultipleBlockSecurityStatus(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wBlockNo,                                                      /**< [In] Block number for which the status should be returned. */
        uint16_t wNoOfBlocks,                                                   /**< [In] Number of blocks to be used for returning the status. */
        uint8_t * pStatus,                                                      /**< [Out] The status of the block number mentioned in wBlockNo until wNumBlocks. */
        uint16_t * pStatusLen                                                   /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a Extended Multiple block fast read command. When receiving the Read Multiple Block command, the VICC shall read the requested
 * block(s) and send back its value in the response. If a VICC supports Extended read multiple blocks command, it shall also support Read multiple
 * blocks command for the first 256 blocks of memory.
 *
 * If the Option_flag (bOption = PHAL_ICODE_OPTION_ON) is set in the request, the VICC shall return the block security status, followed by the block
 * value sequentially block by block. If it is not set (bOption = PHAL_ICODE_OPTION_OFF), the VICC shall return only the block value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ExtendedFastReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *      \arg If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
         *           is available. Format will be 4 byte data 1, 4 byte data 2,  4 byte data N
         *      \arg If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
         *           Format of the response will be Status 1, 4 byte data 1, Status 2, 4 byte data 2,  Status N,
         *           4 byte data N
         *           Where 1, 2  N is the block number.
         *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
         */
        uint16_t wBlockNo,                                                      /**< [In] Block number from where the data to be read. */
        uint16_t wNumBlocks,                                                    /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC in with respect to bOption parameter information. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes.*/
);

/** end of phalICode_Optional
 * @}
 */

/** \defgroup phalICode_Custom Commands_Custom
 * \brief This component implements the custom commands of ICode products.
 * @{
 */

/**
 * \brief This command enables the EAS mode if the EAS mode is not locked. If the EAS mode is password protected
 * the EAS password has to be transmitted before with \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_SetEAS(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
);

/**
 * \brief This command disables the EAS mode if the EAS mode is not locked. If the EAS mode is password protected
 * the EAS password has to be transmitted before with \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ResetEAS(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
);

/**
 * \brief This command locks the current state of the EAS mode and the EAS ID. If the EAS mode is password protected
 * the EAS password has to be transmitted before with \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_LockEAS(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
);

/**
 * \brief This command returns the EAS sequence if the EAS mode is enabled.
 *
 * \li \c bOption disabled: \c bEasIdMaskLength and \c pEasIdValue are not transmitted, EAS Sequence is returned;
 * \li \c bOption enabled and \c bEasIdMaskLength = \c 0: EAS ID is returned;
 * \li \c bOption enabled and \c bEasIdMaskLength > \c 0: EAS Sequence is returned by ICs with matching \c pEasIdValue;
 *
 * If the EAS mode is disabled, the label remains silent.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_EASAlarm(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
         *          \arg #PHAL_ICODE_OPTION_OFF \n
         *                  \c EAS ID mask length and EAS ID value shall not be transmitted. \n
         *                  \c If the EAS mode is enabled, the EAS response is returned from the ICODE IC.
         *                     This configuration is compliant with the EAS command of the ICODE IC
         *          \arg #PHAL_ICODE_OPTION_ON. \n
         *                  \c Within the command the EAS ID mask length has to be transmitted to identify how
         *                     many bits of the following EAS ID value are valid (multiple of 8-bits). Only those
         *                     ICODE ICs will respond with the EAS sequence which have stored the corresponding
         *                     data in the EAS ID configuration (selective EAS) and if the EAS Mode is set. \n
         *                  \c If the EAS ID mask length is set to 0, the ICODE IC will answer with its EAS ID
         */
        uint8_t * pEasIdValue,                                                  /**< [In] EAS ID; 0, 8 or 16 bits; optional. */
        uint8_t bEasIdMaskLen,                                                  /**< [In] 8 bits; optional. */
        uint8_t ** ppEas,                                                       /**< [Out] EAS ID (16 bits) or EAS Sequence (256 bits). */
        uint16_t * pEasLen                                                      /**< [Out] Length of bytes available in ppEas buffer. */
);

/**
 * \brief This command enables the password protection for EAS. The EAS password has to be transmitted before with
 * \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_PasswordProtectEAS(
        void * pDataParams                                                      /**< [In] Pointer to this layer's parameter structure. */
);

/**
 * \brief This command enables the password protection for AFI. The AFI password has to be transmitted before with
 * \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_PasswordProtectAFI(
        void * pDataParams                                                      /**< [In] Pointer to this layer's parameter structure. */
);

/**
 * \brief With this command, a new EAS identifier is stored in the corresponding configuration memory. If the EAS mode
 * is password protected the EAS password has to be transmitted before with \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WriteEASID(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t * pEasIdValue                                                   /**< [In] EAS ID; 16 bits. */
);

/**
 * \brief Performs ReadEPC command. On this command, the label will respond with it's EPC data.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadEPC(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t ** ppEpc,                                                       /**< [Out] EPC data; 96 bits. */
        uint16_t * pEpcLen                                                      /**< [Out] Length of bytes available in ppEpc buffer. */
);

/**
 * \brief Perform GetNXPSystemInformation command. This command allows for retrieving the NXP system information value from the VICC.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetNXPSystemInformation(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppSystemInfo,                                                /**< [Out] The NXP system information of the VICC. */
        uint16_t * pSystemInfoLen                                               /**< [Out] Number of received data bytes. */
);

/**
 * \brief Performs a GetRandomNumber command. On this command, the label will respond with a random number.
 * The received random number shall be used to diversify the password for the \ref phalICode_SetPassword command.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_GetRandomNumber(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppRnd,                                                       /**< [Out] Random number; 16 bits. */
        uint16_t * pRndLen                                                      /**< [Out] Number of bytes in ppRnd buffer. */
);

/**
 * \brief Perforns SetPassword command. With this command the different passwords can be transmitted to the label.
 *
 * This command has to be executed just once for the related passwords if the label is powered.
 *
 * \verbatim
 * [XOR password calculation example]
 * pXorPwd[0] = pPassword[0] ^ pRnd[0];
 * pXorPwd[1] = pPassword[1] ^ pRnd[1];
 * pXorPwd[2] = pPassword[2] ^ pRnd[0];
 * pXorPwd[3] = pPassword[3] ^ pRnd[1];
 * \endverbatim
 *
 * \b Remark: This command can only be executed in addressed or selected mode except of Privay Password.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_SetPassword(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPwdIdentifier,                                                 /**< [In] Password Identifier;
         *          \arg #PHAL_ICODE_SET_PASSWORD_READ
         *          \arg #PHAL_ICODE_SET_PASSWORD_WRITE
         *          \arg #PHAL_ICODE_SET_PASSWORD_PRIVACY
         *          \arg #PHAL_ICODE_SET_PASSWORD_DESTROY
         *          \arg #PHAL_ICODE_SET_PASSWORD_EAS
         */
        uint8_t * pXorPwd                                                       /**< [In] XOR Password; 32 bits. */
);

/**
 * \brief Performs WritePassword command. With this command, a new password is written into the related memory. Note that the
 * old password has to be transmitted before with \ref phalICode_SetPassword. The new password takes effect immediately which
 * means that the new password has to be transmitted with \ref phalICode_SetPassword to get access to protected blocks/pages.
 * \b Remark: This command can only be executed in addressed or selected mode.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WritePassword(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPwdIdentifier,                                                 /**< [In] Password Identifier;
         *          \arg #PHAL_ICODE_SET_PASSWORD_READ
         *          \arg #PHAL_ICODE_SET_PASSWORD_WRITE
         *          \arg #PHAL_ICODE_SET_PASSWORD_PRIVACY
         *          \arg #PHAL_ICODE_SET_PASSWORD_DESTROY
         *          \arg #PHAL_ICODE_SET_PASSWORD_EAS
         */
        uint8_t * pPwd                                                          /**< [In] Plain Password; 32 bits. */
);

/**
 * \brief Performs LockPassword command. This command locks the addressed password. Note that the addressed password
 * has to be transmitted before with \ref phalICode_SetPassword. A locked password can not be changed any longer.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_LockPassword(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPwdIdentifier                                                  /**< [In] Password Identifier;
         *          \arg #PHAL_ICODE_SET_PASSWORD_READ
         *          \arg #PHAL_ICODE_SET_PASSWORD_WRITE
         *          \arg #PHAL_ICODE_SET_PASSWORD_PRIVACY
         *          \arg #PHAL_ICODE_SET_PASSWORD_DESTROY
         *          \arg #PHAL_ICODE_SET_PASSWORD_EAS
         */
);

/**
 * \brief Performs Page protection command. This command changes the protection status of a page. Note that the related
 * passwords have to be transmitted before with \ref phalICode_SetPassword if the page is not public.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ProtectPage(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPPAdd_PageNo,                                                  /**< [In] Page number to be protected in case of products that do not have pages
                                                                                 *        charactersized as high and Low. \n
                                                                                 *        Block number to be protected in case of products that have pages
                                                                                 *        charactersized as high and Low.
                                                                                 */
        uint8_t bProtectionStatus                                               /**< [In] Protection status options for the products that do not have pages
                                                                                 *        charactersized as high and Low.
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_PUBLIC
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE
                                                                                 *
                                                                                 *        Extended Protection status options for the products that have pages
                                                                                 *        charactersized as high and Low.
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_READ_LOW
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_WRITE_LOW
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_READ_HIGH
                                                                                 *          \arg #PHAL_ICODE_PROTECT_PAGE_WRITE_HIGH
                                                                                 */
    );

/**
 * \brief Perform LockPageProtectionCondition command. This command permanenty locks the protection status of a page.
 * Note that the related passwords have to be transmitted before with \ref phalICode_SetPassword if the page is not public.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_LockPageProtectionCondition(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPageNo                                                         /**< [In] Page number to be protected. */
);

/**
 * \brief Perform GetMultipleBlockProtectionStatus command. This instructs the label to return the block protection
 * status of the requested blocks.
 *
 * \b Remark: If \c bBlockNo + \c bNoOfBlocks exceeds the total available number of user blocks, the number of received
 * status bytes is less than the requested number. This means that the last returned status byte corresponds to the
 * highest available user block.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 * Note: The memory should be created and sent to the ProtectionStates buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_GetMultipleBlockProtectionStatus(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bBlockNo,                                                       /**< [In] First Block number. */
        uint8_t bNoOfBlocks,                                                    /**< [In] First Block number. */
        uint8_t * pProtectionStates,                                            /**< [Out] Protection states of requested blocks. */
        uint16_t * pNumReceivedStates                                           /**< [Out] Number of received block protection states. */
);

/**
 * \brief Performs Destroy command. This command permanently destroys the label.
 *
 * The Destroy password has to be transmitted before with \ref phalICode_SetPassword.
 * \b Remark: This command is irreversible and the label will never respond to any command again.
 * \b Remark: This command can only be executed in addressed or selected mode.
 *
 * Note: This command is not valid for ICode Dna product as the Destroy feature is part of Mutual
 * Authentication command (refer \ref phalICode_AuthenticateMAM).
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_Destroy(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t * pXorPwd                                                       /**< [In] XOR Password; 32 bits. Pass the password for the ICODE products that
                                                                                 *        supports and NULL for the products that do not support.
                                                                                 */
    );

/**
 * \brief Performs EnablePrivacy command. This command instructs the label to enter privacy mode.
 *
 * In privacy mode, the label will only respond to \ref phalICode_GetRandomNumber and \ref phalICode_SetPassword commands.
 * To get out of the privacy mode, the Privacy password has to be transmitted before with \ref phalICode_SetPassword.
 *
 * Note: This command is not valid for ICode Dna product as the privacy enabling disabling feature is part of Mutual
 * Authentication command (refer \ref phalICode_AuthenticateMAM).
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_EnablePrivacy(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t * pXorPwd                                                       /**< [In] XOR Password; 32 bits. Pass the password for the iCode products
                                                                                 *        that supports and NULL for the products that do not support.
                                                                                 */
    );

/**
 * \brief Perform 64-BitPasswordProtection command. This instructs the label that both of the Read and Write passwords
 * are required for protected access.
 *
 * Note that both the Read and Write passwords have to be transmitted before with \ref phalICode_SetPassword.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_64BitPasswordProtection(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
    );

/**
 * \brief Performs ReadSignature command. On this command, the label will respond with the signature value.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadSignature(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppSign,                                                      /**< [Out] The originality signature returned by the VICC. */
        uint16_t * pSignLen                                                     /**< [Out] Length of originality signature buffer. */
);

/**
 * \brief Reads a multiple 4 byte(s) data from the mentioned configuration block address. Here the starting address of the
 * configuration block should be given in the parameter wBlockAddr and the number of blocks to read from the starting
 * block should be given in the parameter bNoOfBlocks.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockAddr,                                                     /**< [In] Configuration block address. */
        uint8_t bNoOfBlocks,                                                    /**< [In] The n block(s) to read the configuration data. */
        uint8_t ** ppData,                                                      /**< [Out] Multiple of 4 (4u * No Of Blocks) byte(s) of data read from the mentioned
         *         configuration block address.
         */
        uint16_t * pDataLen                                                     /**< [Out]  Number of received configuration data bytes. */
);

/**
 * \brief Writes a 4 byte data to the mentioned configuration block address.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WriteConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag as per ISO15693; \arg #PHAL_ICODE_OPTION_OFF \arg #PHAL_ICODE_OPTION_ON. */
        uint8_t bBlockAddr,                                                     /**< [In] Configuration block address. */
        uint8_t * pData                                                         /**< [In] A 4 byte data to be written to the mentioned configuration block address. */
);

/**
 * \brief Enables the random ID generation in the tag. This interfaces is used to instruct the tag to generate
 * a random number in privacy mode.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_PickRandomID(
        void * pDataParams                                              /**< [In] Pointer to this layer's parameter structure. */
);

/**
 * \brief Provides the tag tamper status.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface
 *
 * Response will contain the below information.
 *  - Byte1: TagTamper Current State
 *  - Byte2: TagTamper Stored State
 *  | **TT Current State**                       | **TT Stored State**                        |
 *  |:-------------------------------------------|:-------------------------------------------|
 *  | O: Open                                    | O: Open                                    |
 *  | C: Closed                                  | C: Closed                                  |
 *  | I: Invalid                                 | E: Electrical Error                        |
 *  | E: Electrical Error                        | Z: Backup Management Error - Invalid Data  |
 *  | 0: Feature not active or info not provided | 0: Feature not active or info not provided |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If any of the DataParams are null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadTT(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag as per ISO15693
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON.
                                                                                 */
        uint8_t ** ppResponse,                                                  /**< [Out] Two bytes of Tag Tamper status. */
        uint16_t * pRspLen                                                      /**< [Out]  Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Performs Parameter Request command. When receiving VICC PARAMETER REQUEST, NTAG5 I2C returns all supported bit rates
 * and timing information.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ParameterRequest(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pBitRate,                                                     /**< [Out] One byte buffer containing the supported bitrates.
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_26KBPS_BOTH_DIRECTIONS
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD
                                                                                 */
        uint8_t * pTiming                                                       /**< [Out] One byte buffer containing the supported timing information.
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_320_9_US
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_160_5_US
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_80_2_US
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_SAME_BOTH_DIRECTIONS
                                                                                 */
    );

/**
 * \brief Performs Parameter Select command. PARAMETER SELECT command is used to activate one bit rate combination and the T1
 * timing indicated in PARAMETER REQUEST response. Only one option in each direction shall be chosen. After the response to PARAMETER
 * SELECT command, new parameters are valid.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ParameterSelect(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bBitRate,                                                       /**< [In] One byte buffer containing the supported bitrates.
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_26KBPS_BOTH_DIRECTIONS
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD
                                                                                 */
        uint8_t bTiming                                                         /**< [In] One byte buffer containing the supported timing information.
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_320_9_US
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_160_5_US
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_80_2_US
                                                                                 *          \arg #PHAL_ICODE_PARAMETERS_TIMING_SAME_BOTH_DIRECTIONS
                                                                                 */
    );

/**
 * \brief Performs a SRAM Read command.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ReadSRAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
                                                                                 *      \arg If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
                                                                                 *           is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
                                                                                 *      \arg If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
                                                                                 *           Format of the response will be Status 1, 4 byte data 1, Status 2, 4 byte data 2, ... Status N,
                                                                                 *           4 byte data N
                                                                                 *           Where 1, 2 ... N is the block number.
                                                                                 *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC. */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
    );

/**
 * \brief Performs a SRAM Write command.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_WriteSRAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
                                                                                 *      \arg If #PHAL_ICODE_OPTION_OFF, The VICC shall return its response when it has completed the write operation
                                                                                 *           starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *           of 32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *      \arg If #PHAL_ICODE_OPTION_ON, The VICC shall wait for the reception of an EOF from the VCD and upon such reception
                                                                                 *           shall return its response.
                                                                                 *      \arg If #PHAL_ICODE_OPTION_DEFAULT, Same as #PHAL_ICODE_OPTION_OFF
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data should be written. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to be written. */
        uint8_t * pData,                                                        /**< [In] Information to be written to VICC. */
        uint16_t wDataLen                                                       /**< [In] Number of data bytes to be written. */
    );

/**
 * \brief Performs a I2CM Read command. This command is used to read from any I2C slave connected to NTAG5 I2C Host.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_I2CMRead (
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bI2CParam,                                                      /**< [In] I2C Slave address from which the data should be read and the information
                                                                                 *        to set the Stop bit.
                                                                                 *          Bits 0 - 6: 7-bit I2C slave address of connected slave
                                                                                 *          Bit 7     : Disable STOP condition
                                                                                 *                      0b: STOP condition will be generated at the end of transaction
                                                                                 *                      1b: STOP condition will be omitted at the end of transaction
                                                                                 */
        uint16_t wDataLen,                                                     /**< [In] Total Number of data bytes to be read. If 1 byte has to be read then the
                                                                                 *        length will be 1.
                                                                                 */
        uint8_t * pData                                                         /**< [Out] Information to be read from the VICC. */
    );

/**
 * \brief Performs a I2CM Write command. This command is used to write to any I2C slave connected to NTAG5 I2C Host.
 *
 * Flag can be set by using \ref phalICode_SetConfig command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_I2CMWrite(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bI2CParam,                                                      /**< [In] I2C Slave address from which the data should be read and the information
                                                                                 *        to set the Stop bit.
                                                                                 *          Bits 0 - 6: 7-bit I2C slave address of connected slave
                                                                                 *          Bit 7     : Disable STOP condition
                                                                                 *                      0b: STOP condition will be generated at the end of transaction
                                                                                 *                      1b: STOP condition will be omitted at the end of transaction
                                                                                 */
        uint8_t * pData,                                                        /**< [In] Information to be written to the VICC. */
        uint16_t wDataLen                                                       /**< [In] Total Number of data bytes to be written. If 1 byte has to be written then the
                                                                                 *        length will be 1.
                                                                                 */
    );

/** end of phalICode_Custom
 * @}
 */

/** \defgroup phalICode_Utils Utilities
 * \brief This component implements the special commands of ease of use. These commands are actually not of
 * ISO15693 or any of the ICode product families. these are generic commands as part of reader library.
 * @{
 */

/**
 * \brief Get the configuration settings.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_GetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration to read.
         *          \arg #PHAL_ICODE_CONFIG_FLAGS
         *          \arg #PHAL_ICODE_CONFIG_ADD_INFO
         *          \arg #PHAL_ICODE_CONFIG_TIMEOUT_US
         *          \arg #PHAL_ICODE_CONFIG_TIMEOUT_MS
         *          \arg #PHAL_ICODE_CONFIG_ENABLE_BUFFERING
         */
        uint16_t * pValue                                                       /**< [In] The value for the mentioned configuration information in wConfig parameter.
         *          \arg #PHAL_ICODE_CONFIG_FLAGS \n
         *              \c #PHAL_ICODE_FLAG_TWO_SUB_CARRIERS \n
         *              \c #PHAL_ICODE_FLAG_DATA_RATE \n
         *              \c #PHAL_ICODE_FLAG_INVENTORY \n
         *              \c #PHAL_ICODE_FLAG_PROTOCOL_EXTENSION \n
         *              \c #PHAL_ICODE_FLAG_SELECTED \n
         *              \c #PHAL_ICODE_FLAG_AFI \n
         *              \c #PHAL_ICODE_FLAG_ADDRESSED \n
         *              \c #PHAL_ICODE_FLAG_NBSLOTS \n
         *              \c #PHAL_ICODE_FLAG_OPTION \n
         *          \n
         *          \arg #PHAL_ICODE_CONFIG_ADD_INFO, any additional information like error code will be avaialble.
         *          \n
         *          \arg #PHAL_ICODE_CONFIG_ENABLE_BUFFERING \n
         *              \c #PHAL_ICODE_DISABLE \n
         *              \c #PHAL_ICODE_ENABLE \n
         */
);

/**
 * \brief Set the configuration settings.
 *
 * \b NOTE: Both the flags #PHAL_ICODE_FLAG_DATA_RATE and #PHAL_ICODE_FLAG_FAST_DATA_RATE should not be combined,
 *          it should be passed seperatly along with other flag.
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_SetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration to update.
         *          \arg #PHAL_ICODE_CONFIG_FLAGS
         *          \arg #PHAL_ICODE_CONFIG_ENABLE_BUFFERING
         */
        uint16_t wValue                                                         /**< [In] The value for the mentioned configuration information in wConfig parameter.
         *          \arg #PHAL_ICODE_CONFIG_FLAGS \n
         *              \c #PHAL_ICODE_FLAG_TWO_SUB_CARRIERS \n
         *              \c #PHAL_ICODE_FLAG_DATA_RATE \n
         *              \c #PHAL_ICODE_FLAG_INVENTORY \n
         *              \c #PHAL_ICODE_FLAG_PROTOCOL_EXTENSION \n
         *              \c #PHAL_ICODE_FLAG_SELECTED \n
         *              \c #PHAL_ICODE_FLAG_AFI \n
         *              \c #PHAL_ICODE_FLAG_ADDRESSED \n
         *              \c #PHAL_ICODE_FLAG_NBSLOTS \n
         *              \c #PHAL_ICODE_FLAG_OPTION \n
         *              \c #PHAL_ICODE_FLAG_FAST_DATA_RATE \n
         *          \n
         *          \arg #PHAL_ICODE_CONFIG_ENABLE_BUFFERING \n
         *              \c #PHAL_ICODE_DISABLE \n
         *              \c #PHAL_ICODE_ENABLE \n
         */
);

/**
 * \brief Get the type of Tag
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_GetTagType(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t * pTagType                                                     /**< [In] The type of ICode tag.
         *          \arg #PHAL_ICODE_TAG_TYPE_UNKNOWN
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLI
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLI_S
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLI_L
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLIX
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLIX_S
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLIX_L
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_SLI_X2
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_DNA
         *          \arg #PHAL_ICODE_TAG_TYPE_ICODE_NTAG5_I2C
         */
);

#endif /* NXPRDLIB_REM_GEN_INTFS */

/** end of phalICode_Utils
 * @}
 */

/** end of phalICode_Generic
 * @}
 */

/** end of phalICode
 * @}
 */
#endif /* NXPBUILD__PHAL_ICODE */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHALICODE_H */
