/*----------------------------------------------------------------------------*/
/* Copyright 2020 NXP                                                         */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** \file
*
* Example Source for NfcrdlibEx10_MIFAREDESFire_EVx that demonstrates communication
* with MIFARE DESFire EV2 contactless IC and MIFARE DESFire EV3 contactless IC cards.
* This example will Detect and activater the Type-A card.
* Will check if card is MIFAREDESFire EV2 and MIFAREDESFire EV3 cards to proceed further with the example.
* If card is MIFAREDESFire EV2 and MIFAREDESFire EV3 then it will format the card and create application with KEYTYPE selected.
* After creating application, it will create one Standard File and one Transaction MAC File. It will
* demo read and write operations with the card and will show-case commit transaction if the selected
* KEYTYPE is AES.
* Please refer Readme.txt file for Hardware Pin Configuration, Software Configuration and steps to build and
* execute the project which is present in the same project directory.
*
* $Author$
* $Revision$ (v06.10.00)
* $Date$
*
*/

/**
* Reader Library Headers
*/
#include <phApp_Init.h>
#include <NfcrdlibEx10_MIFAREDESFire_EVx.h>


/* *****************************************************************************************************************
* MACROS/Defines
* ***************************************************************************************************************** */
#define PH_EXMFCRYPTO_MFDFCRYPTO_MIFAREDESFIRE_SAK      0x20

#define DES_KEY_ADDRESS         10       /* PICC Key entry number in key store. */
#define DES_KEY_VERSION         00       /* PICC Key entry number in key store. */

#define DES3K3_KEY_ADDRESS      11       /* PICC Key entry number in key store. */
#define DES3K3_KEY_VERSION      00       /* PICC Key entry number in key store. */

#define AES128_KEY_ADDRESS      12       /* PICC Key entry number in key store. */
#define AES128_KEY_VERSION      00       /* PICC Key entry number in key store. */

#define STDDATAFILE1             1       /* File number of Standard data file 1. */
#define TMAC_FILE                2       /* File number of Transaction MAC file 1. */

#define PICC_MASTER_KEY          0
#define APP_MASTER_KEY           0

#define KEYCOUNT                21
#define KEYVERSIONS              1

#define SELECT_DES_KEY           1
#define SELECT_DES3K3_KEY        2
#define SELECT_AES_KEY           3
#define SELECT_BREAK             4

/*******************************************************************************
**   Global Variable Declaration
*******************************************************************************/

phacDiscLoop_Sw_DataParams_t       * pDiscLoop;     /* Pointer to Discovery loop data-params */
phpalI14443p3a_Sw_DataParams_t     * ppalI14443p3a; /* Pointer to PAL 14443-3A data-params */
phpalI14443p4_Sw_DataParams_t      * ppalI14443p4;  /* Pointer to PAL 14443-4 data-params */
phKeyStore_Sw_DataParams_t         * pKeyStore;     /* Pointer to Keystore data-params */
phalMfdfEVx_Sw_DataParams_t        * palMfdfEVx;    /* Pointer to MIFAREDESFire Ev2 data-params */


phKeyStore_Sw_KeyEntry_t       keyEntry[KEYCOUNT];
phKeyStore_Sw_KeyVersionPair_t keyVersion[KEYCOUNT * KEYVERSIONS];
phKeyStore_Sw_KUCEntry_t       keyUsage[KEYCOUNT];

#ifdef PHOSAL_FREERTOS_STATIC_MEM_ALLOCATION
uint32_t MfDesTaskBuffer[MIFAREDESFIRE_TASK_STACK];
#else /* PHOSAL_FREERTOS_STATIC_MEM_ALLOCATION */
#define MfDesTaskBuffer    NULL
#endif /* PHOSAL_FREERTOS_STATIC_MEM_ALLOCATION */

static uint8_t pISOFileName[16];
static uint8_t bISOFileLen;

/* AES PICC key */
static uint8_t aPICC_MasterKey_DES[16] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t aPICC_MasterKey_3K3DES[24] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t aAES128Key[24] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* AFC APPLICATION id */
static uint8_t bAfcApp[3] = {0x2A, 0x8C, 0xF1};

/* This 33 bytes string is used for Originality check.
* NOTE: This should be provided by product team once the key is finalized.
*/
const char *publickey_str = "046F70AC557F5461CE5052C8E4A7838C11C7A236797E8A0730A101837C004039C";

/* *****************************************************************************************************************
* Types/Structure Declarations
* ***************************************************************************************************************** */

/* *****************************************************************************************************************
* Function Prototypes
* ***************************************************************************************************************** */
static void ConfigureDiscoveryLoop(void);
phStatus_t Test_MIFAREDESFireEV2_Features();

/*******************************************************************************
**   Function Definitions
*******************************************************************************/
/* Utility function to print a buffer */
void PRINT_BUFFER(uint8_t * pBuffer, uint16_t wLength)
{
    uint16_t wIndex;

    for (wIndex = 0; wIndex < wLength; ++wIndex)
    {
        DEBUG_PRINTF("%02X ", pBuffer[wIndex]);
    }
    DEBUG_PRINTF("\n");
}

phStatus_t StoreKeysInKeyStore(phKeyStore_Sw_DataParams_t * pKeyStore)
{
    phStatus_t status;

    /* Set the default PICC Master key in the key store */
    status = phKeyStore_FormatKeyEntry(pKeyStore, DES_KEY_ADDRESS, PH_CRYPTOSYM_KEY_TYPE_DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(pKeyStore, DES_KEY_ADDRESS, 0x00, PH_CRYPTOSYM_KEY_TYPE_DES,
            aPICC_MasterKey_DES, DES_KEY_VERSION);
    CHECK_SUCCESS(status);

    status = phKeyStore_FormatKeyEntry(pKeyStore, DES3K3_KEY_ADDRESS, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(pKeyStore, DES3K3_KEY_ADDRESS, 0x00, PH_CRYPTOSYM_KEY_TYPE_3K3DES,
            aPICC_MasterKey_3K3DES, DES3K3_KEY_VERSION);
    CHECK_SUCCESS(status);

    status = phKeyStore_FormatKeyEntry(pKeyStore, AES128_KEY_ADDRESS, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    return phKeyStore_SetKeyAtPos(pKeyStore, AES128_KEY_ADDRESS, 0x00, PH_CRYPTOSYM_KEY_TYPE_AES128,
            aAES128Key, AES128_KEY_VERSION);
}

/* Create application */
phStatus_t Create_Application(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx, uint16_t wKeyNo, uint16_t wKeyVer)
{
    phStatus_t status;
    uint8_t bOption;
    uint8_t bKeySettings1;
    uint8_t bKeySettings2;
    uint8_t bKeySettings3;
    uint8_t bKeySetValues[4];
    uint8_t bISOFileId[2];

    DEBUG_PRINTF("Performing App Creation ...... \n");

    /* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
    bOption = 0x00;

    /* Iso File Id = 0x00AA. Sent LSB first */
    bISOFileId[0] = 0x12;
    bISOFileId[1] = 0x34;

    /* DF Name = AFCAPPLICATION */
    bISOFileLen = 14;
    memcpy(pISOFileName, "AFCAPPLICATION", bISOFileLen);

    /* Application masterkey settings 00001011
    * =  conf. changeable +
    *    create/delete file requires authentication +
    *   directory access is free +
    *   App MK is changeable
    *   Change access rights = 0x00 (AMK)
    */
    bKeySettings1 = 0x0F;

    /* Keysettings 2 = 10100110
    * AES crypto + supports 2 byte ISO fids +
    * max 6 keys in this app
    */
    if (wKeyNo == DES_KEY_ADDRESS)
    {
        bKeySettings2 = 0x02;
    }
    else if (wKeyNo == DES3K3_KEY_ADDRESS)
    {
        bKeySettings2 = 0x42;
    }
    else
    {
        bKeySettings2 = 0x82;
    }

    /* Key Set Values */
    bKeySetValues[0] = 0xAA;         /* byte0 = AKS ver */
    bKeySetValues[1] = 0x00;         /* byte1 = #keysets */
    bKeySetValues[2] = 0x00;         /* byte2 = MaxKeysize */
    bKeySetValues[3] = 0x00;         /* byte3 = AppKeySetSettings */
    bKeySettings3 = 0x00;

    /* Create AFC Application */
    status = phalMfdfEVx_CreateApplication( pAlMfdfEVx, bOption, bAfcApp, bKeySettings1, bKeySettings2,
            bKeySettings3, &bKeySetValues[0], bISOFileId, pISOFileName, bISOFileLen);
    CHECK_SUCCESS(status);

    DEBUG_PRINTF("\tApp creation DONE! \n");

    /* Select Application */
    DEBUG_PRINTF("Performing App Selection ...... \n");
    status = phalMfdfEVx_SelectApplication(pAlMfdfEVx, 0x00, bAfcApp, NULL);
    CHECK_SUCCESS(status);

    /* Authenticate the Application. */
    if (wKeyNo == DES_KEY_ADDRESS)
    {
        status = phalMfdfEVx_Authenticate(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, wKeyNo,
                wKeyVer, APP_MASTER_KEY, NULL, 0);
        CHECK_SUCCESS(status);
    }
    else if (wKeyNo == DES3K3_KEY_ADDRESS)
    {
        status = phalMfdfEVx_AuthenticateISO(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, wKeyNo,
                wKeyVer, APP_MASTER_KEY, NULL, 0);
        CHECK_SUCCESS(status);
    }
    else
    {
        status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, wKeyNo,
                        wKeyVer, APP_MASTER_KEY, NULL, 0);
        CHECK_SUCCESS(status);
    }

    DEBUG_PRINTF("\tApp Selection DONE! \n");

    return 0;
}

/* Create a standard data file */
phStatus_t CreateStandardFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx, uint16_t wKeyNo, uint16_t wKeyVer)
{
    phStatus_t status;
    uint8_t IsoFileID[2] = {0x56, 0x78};
    uint8_t fileOption = 0x00;
    uint8_t accessRights[2] = {0x00, 0x00};
    uint8_t fileSize[3] = {0xFF, 0x00, 0x00};

    status  = phalMfdfEVx_CreateStdDataFile(pAlMfdfEVx, 0X00, STDDATAFILE1, IsoFileID, fileOption, accessRights,
        fileSize);
    CHECK_SUCCESS(status);

    /* Authenticate the file */
    if (wKeyNo == DES_KEY_ADDRESS)
    {
        status = phalMfdfEVx_Authenticate(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, wKeyNo,
                wKeyVer, APP_MASTER_KEY, NULL, 0);
        CHECK_SUCCESS(status);
    }
    else if (wKeyNo == DES3K3_KEY_ADDRESS)
    {
        status = phalMfdfEVx_AuthenticateISO(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, wKeyNo,
                wKeyVer, APP_MASTER_KEY, NULL, 0);
        CHECK_SUCCESS(status);
    }
    else
    {
        status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, wKeyNo,
                wKeyVer, APP_MASTER_KEY, NULL, 0);
        CHECK_SUCCESS(status);
    }

    DEBUG_PRINTF("***** Creating standard data file SUCCESS!!*******\n");

    return status;
}

/* Create a Transaction MAC file */
phStatus_t CreateTMACFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
    phStatus_t status;
    uint8_t accessRights[2] = {0xF0, 0x0F};

    status = phalMfdfEVx_CreateTransactionMacFile(pAlMfdfEVx, 0xFFFF, TMAC_FILE, 0x00, accessRights,
            AES128_KEY_ADDRESS, 0x02, aAES128Key, AES128_KEY_VERSION, NULL, 0);
    CHECK_SUCCESS(status);

    DEBUG_PRINTF("***** Creating Transaction MAC file SUCCESS!!*******\n");

    return status;
}

phStatus_t ReadData(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
    phStatus_t status;
    uint8_t offset[3] = {0x00, 0x00, 0x00};
    uint8_t length[3] = {0x64, 0x00, 0x00};
    uint8_t * pRxData = NULL;
    uint16_t wRxLen = 0;
    uint16_t wIndex = 0;

    status = phalMfdfEVx_ReadData(pAlMfdfEVx, 0x00, 0x00, STDDATAFILE1, &offset[0], &length[0], &pRxData, &wRxLen);
    CHECK_SUCCESS(status);

    DEBUG_PRINTF("***** Reading Data from Standard Data File Successful *****\n");

    for(wIndex = 0; wIndex < wRxLen; wIndex++)
    {
        if((wIndex % 16) == 0)
        {
            DEBUG_PRINTF("\n");
        }

        DEBUG_PRINTF("%02x ", pRxData[wIndex]);
    }
    DEBUG_PRINTF("\n ");

    return 0;
}

phStatus_t WriteData(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
    phStatus_t status;
    uint8_t data[100] = {0};
    uint8_t offset[3] = {0x00, 0x00, 0x00};
    uint8_t length[3] = {0x64, 0x00, 0x00};
    uint16_t wIndex = 0;

    for(wIndex = 0; wIndex < sizeof(data); wIndex++)
    {
        data[wIndex] = wIndex;
    }

    status = phalMfdfEVx_WriteData(pAlMfdfEVx, 0x00, 0x00, STDDATAFILE1, &offset[0], &data[0], &length[0]);
    CHECK_SUCCESS(status);

    DEBUG_PRINTF("***** Writing Data to Standard Data File Successful *****\n");

    return 0;
}

phStatus_t CommitTransaction(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
    phStatus_t status;
    uint8_t aTMC[4] = {0};
    uint8_t aTMV[8] = {0};
    uint8_t bIndex = 0;

    /* Perform Commit Transaction. */
    status = phalMfdfEVx_CommitTransaction(pAlMfdfEVx, 0x01, aTMC, aTMV);
    CHECK_SUCCESS(status);

    DEBUG_PRINTF ("TMC: " );
    for (bIndex = 0; bIndex < sizeof(aTMC); bIndex++)
    {
        DEBUG_PRINTF ("%02X ", aTMC[bIndex] );
    }
    DEBUG_PRINTF("\n");

    DEBUG_PRINTF ("TMV: " );
    for (bIndex = 0; bIndex < sizeof(aTMV); bIndex++)
    {
        DEBUG_PRINTF ("%02X ", aTMV[bIndex] );
    }
    DEBUG_PRINTF("\n");

    return status;
}

phStatus_t DataReadWrite(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx,  uint16_t wKeyNo, uint16_t wKeyVer)
{
    phStatus_t status;

    /* Authenticate PICC Master Application. */
    status = phalMfdfEVx_Authenticate(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, DES_KEY_ADDRESS,
            DES_KEY_VERSION, PICC_MASTER_KEY, NULL, 0);
    CHECK_SUCCESS(status);

    /* Format Application. */
    status = phalMfdfEVx_Format(pAlMfdfEVx);
    CHECK_SUCCESS(status);

    /* Create Application. */
    status = Create_Application(pAlMfdfEVx , wKeyNo, wKeyVer);
    CHECK_SUCCESS(status);

    /* Create TransactionMAC file */
    if(wKeyNo == AES128_KEY_ADDRESS)
    {
        status = CreateTMACFile(pAlMfdfEVx);
        CHECK_SUCCESS(status);
    }

    /* CreateStandardFile. */
    status = CreateStandardFile(pAlMfdfEVx, wKeyNo, wKeyVer);
    CHECK_SUCCESS(status);

    /* Read the Data. */
    status = ReadData(pAlMfdfEVx);
    CHECK_SUCCESS(status);

    /* Write the data. */
    status = WriteData(pAlMfdfEVx);
    CHECK_SUCCESS(status);

    /* Read the Data. */
    status = ReadData(pAlMfdfEVx);
    CHECK_SUCCESS(status);

    /* Commit Transaction. */
    if(wKeyNo == AES128_KEY_ADDRESS)
    {
        status = CommitTransaction(pAlMfdfEVx);
        CHECK_SUCCESS(status);
    }
    return 0;
}


/*******************************************************************************
**   Main Function
*******************************************************************************/
int main (void)
{
    do
    {
        phStatus_t            status = PH_ERR_INTERNAL_ERROR;
        phNfcLib_Status_t     dwStatus;
#ifdef PH_PLATFORM_HAS_ICFRONTEND
        phNfcLib_AppContext_t AppContext = {0};
#endif /* PH_PLATFORM_HAS_ICFRONTEND */

#ifndef PH_OSAL_NULLOS
        phOsal_ThreadObj_t MfDes;
#endif /* PH_OSAL_NULLOS */

        /* Perform Controller specific initialization. */
        phApp_CPU_Init();

        /* Perform OSAL Initialization. */
        (void)phOsal_Init();

        /* Print Example application name */
        DEBUG_PRINTF("\n *** MIFARE DESFire EV2 Example *** \n");

#ifdef PH_PLATFORM_HAS_ICFRONTEND
        status = phbalReg_Init(&sBalParams, sizeof(phbalReg_Type_t));
        CHECK_STATUS(status);

        AppContext.pBalDataparams = &sBalParams;
        dwStatus = phNfcLib_SetContext(&AppContext);
        CHECK_NFCLIB_STATUS(dwStatus);
#endif /* PH_PLATFORM_HAS_ICFRONTEND */

        /* Initialize library */
        dwStatus = phNfcLib_Init();
        CHECK_NFCLIB_STATUS(dwStatus);
        if(dwStatus != PH_NFCLIB_STATUS_SUCCESS) break;

        /* Set the generic pointer */
        pHal = phNfcLib_GetDataParams(PH_COMP_HAL);
        pDiscLoop = (phacDiscLoop_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_AC_DISCLOOP);
        ppalI14443p3a = (phpalI14443p3a_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_PAL_ISO14443P3A);
        ppalI14443p4 = (phpalI14443p4_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_PAL_ISO14443P4);
        pKeyStore = (phKeyStore_Sw_DataParams_t *)  phNfcLib_GetDataParams(PH_COMP_KEYSTORE);
        palMfdfEVx = (phalMfdfEVx_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_AL_MFDFEVX);

        /* Initialize other components that are not initialized by NFCLIB and configure Discovery Loop. */
        status = phApp_Comp_Init(pDiscLoop);
        CHECK_STATUS(status);
        if(status != PH_ERR_SUCCESS) break;

        /* Configure the IRQ */
        status = phApp_Configure_IRQ();
        CHECK_STATUS(status);
        if(status != PH_ERR_SUCCESS) break;

#ifndef PH_OSAL_NULLOS

        MfDes.pTaskName = (uint8_t *) "MfDesEV2";
        MfDes.pStackBuffer = MfDesTaskBuffer;
        MfDes.priority = MIFAREDESFIRE_TASK_PRIO;
        MfDes.stackSizeInNum = MIFAREDESFIRE_TASK_STACK;
        phOsal_ThreadCreate(&MfDes.ThreadHandle, &MfDes, &T_MIFAREDESFireEV2, pDiscLoop);

        /* Start Scheduler */
        phOsal_StartScheduler();

        /* Print Error info */
        DEBUG_PRINTF("RTOS Error : Scheduler exited. \n");

#else
        (void)T_MIFAREDESFireEV2(pDiscLoop);
#endif
    }  while(0);

    while(1); //Comes here if initialization failure or scheduler exit due to error

    return 0;
}

/***********************************************************************************************
* \brief   This function demonstrates the MIFARE DESFire EV2 application as mentioned below
*          1. Performs Type A Card detection and Activation.
*          2. Performs SAK comparison to confirm that the detected Tag is MIFARE DESFire Ev2 and MIFAREDESFire Ev3.
*          3. Format the card and create the new Application with selected KEYTYPE.
*          4. Performs creation of one Standard File and one Transaction MAC file.
*          5. Performs read and write operation.
*          6. If selected KEYTYPE is AES, it will perform commit transaction operation.
*
* \param   *pParams
* \return  This function will never return
**********************************************************************************************/
void T_MIFAREDESFireEV2(void *pParams)
{
    phStatus_t  status = 0;
    uint16_t    wTechDetected = 0;
    uint8_t     bTagType;

    /* Configure Discovery Loop functionality required by this example to only activate Type A cards. */
    ConfigureDiscoveryLoop();

    /* Continuous loop */
    while(1)
    {
        DEBUG_PRINTF("\nReady to detect\n");

        do
        {
            /* Field off */
            status = phhalHw_FieldOff(pHal);
            DEBUG_ERROR_PRINT(status);

            status = phhalHw_Wait(pHal,PHHAL_HW_TIME_MICROSECONDS, 5100);
            DEBUG_ERROR_PRINT(status);

            /* Configure Discovery loop for tag detection */
            status = phacDiscLoop_SetConfig(pDiscLoop, PHAC_DISCLOOP_CONFIG_NEXT_POLL_STATE, PHAC_DISCLOOP_POLL_STATE_DETECTION);
            DEBUG_ERROR_PRINT(status);

            /* Run Discovery loop in poll mode to detect Card. */
            status = phacDiscLoop_Run(pDiscLoop, PHAC_DISCLOOP_ENTRY_POINT_POLL);
        }
        while((status & PH_ERR_MASK) != PHAC_DISCLOOP_DEVICE_ACTIVATED); /* Exit on Card detection */

        /* Card detected */
        /* Get the tag types detected info */
        status = phacDiscLoop_GetConfig(pDiscLoop, PHAC_DISCLOOP_CONFIG_TECH_DETECTED, &wTechDetected);
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Check for Type A tag detection */
            if (PHAC_DISCLOOP_CHECK_ANDMASK(wTechDetected, PHAC_DISCLOOP_POS_BIT_MASK_A))
            {
                /* Bit b3 is set to zero, [Digital] 4.8.2 */
                /* Mask out all other bits except for b7 and b6 */
                bTagType = (pDiscLoop->sTypeATargetInfo.aTypeA_I3P3[0].aSak & 0x60);
                bTagType = bTagType >> 5;

                /* Check if the detected tag is MIFARE DESFire EV2. */
                if (pDiscLoop->sTypeATargetInfo.aTypeA_I3P3[0].aSak == PH_EXMFCRYPTO_MFDFCRYPTO_MIFAREDESFIRE_SAK)
                {
                    status = Test_MIFAREDESFireEV2_Features();
                    DEBUG_ERROR_PRINT(status);
                }
                else
                {
                    DEBUG_PRINTF("\nThe detected card does not seem to be MIFARE DESFire EV2 Card. Aborting.\n\n");
                }

                DEBUG_PRINTF("\nPlease Remove the Card\n\n");

                /* Field reset */
                status = phhalHw_FieldReset(pHal);
                DEBUG_ERROR_PRINT(status);

                /* Make sure that example application is not detecting the same card continuously. */
                do
                {
                    /* Send WakeUpA */
                    status = phpalI14443p3a_WakeUpA(ppalI14443p3a,
                        pDiscLoop->sTypeATargetInfo.aTypeA_I3P3[0].aAtqa);
                    /* Check for Status */
                    if (status != PH_ERR_SUCCESS)
                    {
                        break; /* Card Removed, break from the loop */
                    }

                    /* Send HaltA */
                    (void)phpalI14443p3a_HaltA(ppalI14443p3a);

                    /* Delay of 5ms*/
                    status = phhalHw_Wait(pHal, PHHAL_HW_TIME_MILLISECONDS, 5);
                    DEBUG_ERROR_PRINT(status);
                }while(1);

                (void)phpalI14443p4_ResetProtocol(ppalI14443p4);
            }
        }
    }
}

phStatus_t Test_MIFAREDESFireEV2_Features(void)
{
    phStatus_t status;
    uint32_t iChoice = 0;

    /* init. keystore */
    status = phKeyStore_Sw_Init(pKeyStore,
        sizeof(phKeyStore_Sw_DataParams_t),
        keyEntry,
        KEYCOUNT,
        keyVersion,
        KEYVERSIONS,
        keyUsage,
        KEYCOUNT);
    CHECK_SUCCESS(status);

    /* Set all the keys in the S/W key store */
    status = StoreKeysInKeyStore(pKeyStore);
    CHECK_SUCCESS(status);

    /* Clear the console. */
    DEBUG_PRINTF("A Ticket is discovered \n");
    DEBUG_PRINTF("Press 1 Data Read Write using D40 (DES Key) Authentication \n");
    DEBUG_PRINTF("Press 2 Data Read Write using ISO (3K3DES Key) Authentication \n");
    DEBUG_PRINTF("Press 3 Data Read Write using AES (AES Key) Authentication with Commit Transaction\n");
    DEBUG_PRINTF("Press 4 to quit \n");
    DEBUG_PRINTF("Enter your option : ");
    scanf("%X", &iChoice);

    /* To See The Complete Log ,right Click on Console Window,Go to Properties->Layout->Screen Buffer Size->999 */
    switch(iChoice)
    {
    case SELECT_DES_KEY:
        status = DataReadWrite(palMfdfEVx, DES_KEY_ADDRESS, DES_KEY_VERSION);
        CHECK_SUCCESS(status);
        break;

    case SELECT_DES3K3_KEY:
        status = DataReadWrite(palMfdfEVx, DES3K3_KEY_ADDRESS, DES3K3_KEY_VERSION);
        CHECK_SUCCESS(status);
        break;

    case SELECT_AES_KEY:
        status = DataReadWrite(palMfdfEVx, AES128_KEY_ADDRESS, AES128_KEY_VERSION);
        CHECK_SUCCESS(status);
        break;

    case SELECT_BREAK:
        break;

    default:
        DEBUG_PRINTF ( "\n\n" );
        DEBUG_PRINTF ( "Invalid option selected.\n" );
        break;
    }

    return PH_ERR_SUCCESS;
}

/*
* Initialize/Configure functionality required by this example.
*/
static void ConfigureDiscoveryLoop(void)
{
    phStatus_t status;

    /* Configure Discovery loop parameters */
    /* Device limit for Type A */
    status = phacDiscLoop_SetConfig(pDiscLoop, PHAC_DISCLOOP_CONFIG_TYPEA_DEVICE_LIMIT, 1);
    CHECK_STATUS(status);

    /* Bail-out on Type A detect */
    status = phacDiscLoop_SetConfig(pDiscLoop, PHAC_DISCLOOP_CONFIG_BAIL_OUT, PHAC_DISCLOOP_POS_BIT_MASK_A);
    CHECK_STATUS(status);
}

#ifdef NXPBUILD__PHHAL_HW_TARGET
/* Stubbed definitions in case TARGET is enabled */
uint8_t  sens_res[2]     = {0x04, 0x00};
uint8_t  nfc_id1[3]      = {0xA1, 0xA2, 0xA3};
uint8_t  sel_res         = 0x40;
uint8_t  nfc_id3         = 0xFA;
uint8_t  poll_res[18]    = {0x01, 0xFE, 0xB2, 0xB3, 0xB4, 0xB5,
                                   0xB6, 0xB7, 0xC0, 0xC1, 0xC2, 0xC3,
                                   0xC4, 0xC5, 0xC6, 0xC7, 0x23, 0x45 };
#endif /* NXPBUILD__PHHAL_HW_TARGET */

/***********************************************************************************************
*                            End Of File
**********************************************************************************************/
