/* Copyright 2021 NXP
*
* This software is owned or controlled by NXP and may only be used
* strictly in accordance with the applicable license terms.  By expressly
* accepting such terms or by downloading, installing, activating and/or
* otherwise using the software, you are agreeing that you have read, and
* that you agree to comply with and are bound by, such license terms.  If
* you do not agree to be bound by the applicable license terms, then you
* may not retain, install, activate or otherwise use the software.
*
*/

/** \file
* Generic phDriver Component of Reader Library Framework.
* $Author$ Petar Hlad
* $Revision$
* $Date$
*
* History:
*  RS: Generated 12. Jul 2021
*
*/

#include <stdint.h>
#include "fsl_gpio.h"
#include "fsl_adapter_gpio.h"
#include "fsl_pit.h"
#include "ph_TypeDefs.h"
#include "phDriver.h"
#include "BoardSelection.h"

/* *****************************************************************************************************************
 * Internal Definitions
 * ***************************************************************************************************************** */
#define IMX_TIMER_MAX_32BIT      0xFFFFFFFFU

/* *****************************************************************************************************************
 * Type Definitions
 * ***************************************************************************************************************** */
static const GPIO_Type *pGpiosBaseAddr[] = GPIO_BASE_PTRS;

static pphDriver_TimerCallBck_t pPitTimerCallBack;
static volatile uint8_t dwTimerExp;

/* *****************************************************************************************************************
 * Private Functions Prototypes
 * ***************************************************************************************************************** */
static void phDriver_PitTimerIsrCallBack(void);

phStatus_t phDriver_PinConfig(uint32_t dwPinNumber, phDriver_Pin_Func_t ePinFunc, phDriver_Pin_Config_t *pPinConfig)
{
    gpio_pin_config_t sGpioConfig;
    uint8_t bPinNum;
    uint8_t bPortGpio;


    if((ePinFunc == PH_DRIVER_PINFUNC_BIDIR) || (pPinConfig == NULL))
    {
        return PH_DRIVER_ERROR | PH_COMP_DRIVER;
    }

    /* Extract the Pin, Gpio, Port details from dwPinNumber */
    bPinNum = (uint8_t)(dwPinNumber & 0xFF);
    bPortGpio = (uint8_t)((dwPinNumber & 0xFF00)>>8);

    CLOCK_EnableClock(kCLOCK_Iomuxc);
    sGpioConfig.direction = (ePinFunc == PH_DRIVER_PINFUNC_OUTPUT)?kGPIO_DigitalOutput:kGPIO_DigitalInput;
    sGpioConfig.outputLogic  =  pPinConfig->bOutputLogic;

    if(ePinFunc == PH_DRIVER_PINFUNC_INTERRUPT)
    {
        sGpioConfig.interruptMode = pPinConfig->eInterruptConfig;

        GPIO_PinInit((GPIO_Type *)pGpiosBaseAddr[bPortGpio],bPinNum,&sGpioConfig);
        GPIO_PortClearInterruptFlags((GPIO_Type *)pGpiosBaseAddr[bPortGpio],1U << bPinNum);
        GPIO_PortEnableInterrupts((GPIO_Type *)pGpiosBaseAddr[bPortGpio],1U << bPinNum);

    }

    else
    {
        sGpioConfig.interruptMode = kGPIO_NoIntmode;
        GPIO_PinInit((GPIO_Type *)pGpiosBaseAddr[bPortGpio],bPinNum,&sGpioConfig);
    }

    return PH_DRIVER_SUCCESS;
}

uint8_t phDriver_PinRead(uint32_t dwPinNumber, phDriver_Pin_Func_t ePinFunc)
{
    uint8_t bValue;
    uint8_t bGpioNum;
    uint8_t bPinNum;

    /* Extract the Pin, Gpio details from dwPinNumber */
    bPinNum = (uint8_t)(dwPinNumber & 0xFF);
    bGpioNum = (uint8_t)((dwPinNumber & 0xFF00)>>8);

    if(ePinFunc == PH_DRIVER_PINFUNC_INTERRUPT)
    {
        bValue = (uint8_t)((GPIO_GetPinsInterruptFlags((GPIO_Type *)pGpiosBaseAddr[bGpioNum]) >> bPinNum) & 0x01);
    }
    else
    {
        bValue = (uint8_t)GPIO_ReadPinInput((GPIO_Type *)pGpiosBaseAddr[bGpioNum], bPinNum);
    }

    return bValue;
}

void phDriver_PinWrite(uint32_t dwPinNumber, uint8_t bValue)
{
    uint8_t bGpioNum;
    uint8_t bPinNum;

    /* Extract the Pin, Gpio details from dwPinNumber */
    bPinNum = (uint8_t)(dwPinNumber & 0xFF);
    bGpioNum = (uint8_t)((dwPinNumber & 0xFF00)>>8);

    GPIO_WritePinOutput((GPIO_Type *)pGpiosBaseAddr[bGpioNum], bPinNum, bValue);
}

void phDriver_PinClearIntStatus(uint32_t dwPinNumber)
{
    uint8_t bGpioNum;
    uint8_t bPinNum;

    /* Extract the Pin, Gpio details from dwPinNumber */
    bPinNum = (uint8_t)(dwPinNumber & 0xFF);
    bGpioNum = (uint8_t)((dwPinNumber & 0xFF00)>>8);

    GPIO_ClearPinsInterruptFlags((GPIO_Type *)pGpiosBaseAddr[bGpioNum], (1U << bPinNum));
}

phStatus_t phDriver_TimerStart(phDriver_Timer_Unit_t eTimerUnit, uint32_t dwTimePeriod, pphDriver_TimerCallBck_t pTimerCallBack)
{
    uint64_t          qwTimerCnt;
    uint32_t          dwTimerFreq;

    dwTimerFreq = CLOCK_GetRootClockFreq(PH_DRIVER_IMX_PIT_CLK);

    /* Timer count = (delay * freq)/Units. */
    qwTimerCnt = dwTimerFreq;
    qwTimerCnt = (qwTimerCnt / eTimerUnit);
    qwTimerCnt = (dwTimePeriod * qwTimerCnt);

    /* 32-bit timers. */
    if(qwTimerCnt > (uint64_t)IMX_TIMER_MAX_32BIT)
    {
        return PH_DRIVER_ERROR | PH_COMP_DRIVER;
    }

    if(pTimerCallBack == NULL)
    {    /* Timer Start is blocking call. */
        dwTimerExp = 0;
        pPitTimerCallBack = phDriver_PitTimerIsrCallBack;
    }
    else
    {   /* Call the Timer callback. */
        pPitTimerCallBack = pTimerCallBack;
    }

    /* Structure of initialize PIT */
    pit_config_t pitConfig;

    /* Get default config */
    PIT_GetDefaultConfig(&pitConfig);

    /* Init pit module */
    PIT_Init(PH_DRIVER_IMX_PIT_TIMER, &pitConfig);

    /* Set timer period for channel 0 */
    PIT_SetTimerPeriod(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL, (uint32_t)qwTimerCnt);

    /* Set timer priority of the interrupt */
    NVIC_SetPriority(PH_DRIVER_IMX_TIMER_NVIC, PH_DRIVER_IMX_TIMER_PRIORITY);

    /* Enable timer interrupts for channel 0 */
    PIT_EnableInterrupts(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL, kPIT_TimerInterruptEnable);

    /* Enable at the NVIC */
    EnableIRQ(PH_DRIVER_IMX_TIMER_NVIC);

    /* Start channel 0 */
    PIT_StartTimer(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL);

    if(pTimerCallBack == NULL)
    {
        /* Block until timer expires. */
        while(!dwTimerExp);
    }

    return PH_DRIVER_SUCCESS;
}

phStatus_t phDriver_TimerStop(void)
{
    PIT_StopTimer(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL);
    PIT_DisableInterrupts(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL, kPIT_TimerInterruptEnable);

    /* Disable at the NVIC */
    DisableIRQ(PH_DRIVER_IMX_TIMER_NVIC);

    return PH_DRIVER_SUCCESS;
}

void PIT1_IRQHandler(void)
{
    /* Clear interrupt flag.*/
    PIT_ClearStatusFlags(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL, kPIT_TimerFlag);
    /* Single shot timer. Stop it. */
    PIT_StopTimer(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL);
    PIT_DisableInterrupts(PH_DRIVER_IMX_PIT_TIMER, PH_DRIVER_IMX_TIMER_CHANNEL, kPIT_TimerInterruptEnable);

    pPitTimerCallBack();
}

static void phDriver_PitTimerIsrCallBack(void)
{
    /* Mark timer as expired */
    dwTimerExp = 1;
}
