/* Copyright 2021 NXP
*
* This software is owned or controlled by NXP and may only be used
* strictly in accordance with the applicable license terms.  By expressly
* accepting such terms or by downloading, installing, activating and/or
* otherwise using the software, you are agreeing that you have read, and
* that you agree to comply with and are bound by, such license terms.  If
* you do not agree to be bound by the applicable license terms, then you
* may not retain, install, activate or otherwise use the software.
*
*/

/** \file
* Generic phDriver Component of Reader Library Framework.
* $Author$ Petar Hlad
* $Revision$
* $Date$
*
* History:
*  RS: Generated 12. Jul 2021
*
*/

#include "phDriver.h"
#include "BoardSelection.h"
#include "phApp_Init.h"

#include "fsl_pit.h"
#include "fsl_iomuxc.h"
#include "fsl_gpio.h"
#include "fsl_common.h"
#include "fsl_lpspi.h"

#ifndef PHDRIVER_IMX_SPI_POLLING
#include <fsl_lpspi_freertos.h>
#endif

#define PHBAL_REG_IMX_SPI_ID               0x0FU       /**< ID for IMX SPI BAL component */

#ifndef PHDRIVER_IMX_SPI_POLLING
lpspi_rtos_handle_t g_masterHandle;
#endif

phStatus_t phbalReg_Init(
                         void * pDataParams,
                         uint16_t wSizeOfDataParams
                         )
{

    if ( (pDataParams == NULL) || (sizeof(phbalReg_Type_t) != wSizeOfDataParams))
    {
        return (PH_DRIVER_ERROR | PH_COMP_DRIVER);
    }

    ((phbalReg_Type_t *)pDataParams)->wId      = PH_COMP_DRIVER | PHBAL_REG_IMX_SPI_ID;
    ((phbalReg_Type_t *)pDataParams)->bBalType = PHBAL_REG_TYPE_SPI;


    lpspi_master_config_t g_masterConfig;
    memset(&g_masterConfig, 0, sizeof(lpspi_master_config_t));

    g_masterConfig.baudRate = PHDRIVER_IMX_SPI_DATA_RATE;
    g_masterConfig.bitsPerFrame = 8U;
    g_masterConfig.cpol = kLPSPI_ClockPolarityActiveHigh;
    g_masterConfig.cpha = kLPSPI_ClockPhaseFirstEdge;
    g_masterConfig.direction = kLPSPI_MsbFirst;

    g_masterConfig.pcsToSckDelayInNanoSec = 1000000000U / g_masterConfig.baudRate * 20U;
    g_masterConfig.lastSckToPcsDelayInNanoSec = 1000000000U / g_masterConfig.baudRate * 20U;
    g_masterConfig.betweenTransferDelayInNanoSec = 1000000000U / g_masterConfig.baudRate * 20U;

    g_masterConfig.whichPcs = kLPSPI_Pcs0;
    g_masterConfig.pcsActiveHighOrLow = kLPSPI_PcsActiveLow;

    g_masterConfig.pinCfg = kLPSPI_SdiInSdoOut;
    g_masterConfig.dataOutConfig = kLpspiDataOutTristate;

    NVIC_SetPriority(PHDRIVER_IMX_SPI_IRQ, LPSPI_IRQ_PRIORITY);

#ifdef PHDRIVER_IMX_SPI_POLLING
    /* Initialize the LPSPI peripheral */
    LPSPI_MasterInit(PHDRIVER_IMX_SPI_MASTER, &g_masterConfig, CLOCK_GetFreqFromObs(PHDRIVER_IMX_SPI_CLK_SRC));
#else
    LPSPI_RTOS_Init(&g_masterHandle, PHDRIVER_IMX_SPI_MASTER, &g_masterConfig, CLOCK_GetFreqFromObs(PHDRIVER_IMX_SPI_CLK_SRC));
#endif

    return PH_DRIVER_SUCCESS;
}

phStatus_t phbalReg_Exchange(
                             void * pDataParams,
                             uint16_t wOption,
                             uint8_t * pTxBuffer,
                             uint16_t wTxLength,
                             uint16_t wRxBufSize,
                             uint8_t * pRxBuffer,
                             uint16_t * pRxLength
                             )
{
    phStatus_t status = PH_DRIVER_SUCCESS;
    uint8_t * pRxBuf;
    status_t lpspiStatus;
    lpspi_transfer_t g_masterXfer;
    uint8_t g_dummyBuffer[RX_BUFFER_SIZE_MAX];

    memset(&g_masterXfer, 0, sizeof(lpspi_transfer_t));

    if(pRxBuffer == NULL)
    {
        pRxBuf = g_dummyBuffer;
    }
    else
    {
        pRxBuf = pRxBuffer;
    }

    if(pTxBuffer == NULL)
    {
        wTxLength = wRxBufSize;
        g_dummyBuffer[0] = 0xFF;
        pTxBuffer = g_dummyBuffer;
    }


    /* Set up the transfer */
    g_masterXfer.txData = pTxBuffer;
    g_masterXfer.rxData = pRxBuf;
    g_masterXfer.dataSize = wTxLength;
    g_masterXfer.configFlags = kLPSPI_MasterPcs0 | kLPSPI_MasterPcsContinuous;

    /* Start transfer */
#ifdef PHDRIVER_IMX_SPI_POLLING
    lpspiStatus = LPSPI_MasterTransferBlocking(PHDRIVER_IMX_SPI_MASTER, &g_masterXfer);
#else
    lpspiStatus = LPSPI_RTOS_Transfer(&g_masterHandle, &g_masterXfer);
#endif
    if (lpspiStatus != kStatus_Success)
    {
        return (PH_DRIVER_FAILURE | PH_COMP_DRIVER);
    }

    if (pRxLength != NULL)
    {
        *pRxLength = wTxLength;
    }

    return status;
}

phStatus_t phbalReg_SetConfig(
                              void * pDataParams,
                              uint16_t wConfig,
                              uint32_t dwValue
                              )
{
#ifdef PERF_TEST
    switch(wConfig)
    {
    case PHBAL_CONFIG_SPI_BAUD:
        dwSpiBaudRate = dwValue;
        break;
    default:
        return (PH_DRIVER_ERROR | PH_COMP_DRIVER);
    }
#endif /* PERF_TEST */
    return PH_DRIVER_SUCCESS;
}

phStatus_t phbalReg_GetConfig(
                              void * pDataParams,
                              uint16_t wConfig,
                              uint32_t * pValue
                              )
{
#ifdef PERF_TEST
    switch(wConfig)
    {
    case PHBAL_CONFIG_SPI_BAUD:
        *pValue = dwSpiBaudRate;
        break;
    default:
        return (PH_DRIVER_ERROR | PH_COMP_DRIVER);
    }
#endif /* PERF_TEST */
    return PH_DRIVER_SUCCESS;
}

