/*
 * Copyright (c) 2008-2016, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef IPROTOCOLDEVICEINTERFACE_H
#define IPROTOCOLDEVICEINTERFACE_H

#include "../types.h"

#include <vector>

namespace CPPrfeReaderInterface
{

    /// <summary>
    /// Interface which is used by the protocol handler to communicate with the reader.
    /// </summary>
    class IProtocolDeviceInterface
    {
    public:
        IProtocolDeviceInterface();

        /// <summary>
        /// Opens the device
        /// </summary>
        /// <returns>true if opening was successful, otherwise false</returns>
        virtual bool open() = 0;

        /// <summary>
        /// Closes the device
        /// </summary>
        /// <returns>true if closing was successful, otherwise false</returns>
        virtual bool close() = 0;

        /// <summary>
        /// Sends the given bytes to the device
        /// </summary>
        /// <param name="data">Data to be sent</param>
        /// <returns></returns>
        virtual bool send(const std::vector<byte>& data) = 0;

        class DataReadListener
        {
        public:
            /// <summary>
            /// Delegate is called if data is received.
            /// </summary>
            /// <param name="data">Received data</param>
            virtual void dataReadHandler(const std::vector<byte>& data) = 0;
        };

        /// <summary>
        /// Sets the data read event handler.
        /// </summary>
        void setDataReadListener(DataReadListener* handler);


    protected:
        /// <summary>
        /// Raises the DataRead event
        /// </summary>
        /// <param name="data">The read data</param>
        void raiseDataReadEvent(const std::vector<byte>& data);

    private:
        DataReadListener*     m_DataRead;
    };
};

#endif // IPROTOCOLDEVICEINTERFACE_H
