/*
 * Copyright (c) 2008-2016, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "CPPrfePURprotocolHandler.h"

using namespace CPPrfeReaderInterface;

CPPrfePURprotocolHandler::CPPrfePURprotocolHandler(IProtocolDeviceInterface* device)
    : CPPrfeProtocolHandler(device)
{
    m_purEventListener = 0;
}

void CPPrfePURprotocolHandler::setEventListener(CPPrfePURprotocolHandler::EventListener *listener)
{
    m_purEventListener = listener;
}


//************************************************************** Settings

/// <summary>
/// Tries to get the post detect READ settings
/// </summary>
/// <param name="on">Is post detect READ on?</param>
/// <param name="memBank">Target Memory Bank</param>
/// <param name="address">Target Address</param>
/// <param name="size">Target Size</param>
/// <param name="password">Password</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::getPostDetectReadParams(bool& on, byte& memBank, ushort& address, byte& size, std::vector<byte>& password)
{
    on = false;
    memBank = 0;
    address = 0;
    size = 0;
    password.clear();

    std::vector<byte> buffer;
    if (!getParam(0x0006, buffer))
        return false;

    if (buffer.size() != 9)
        return false;

    on = (buffer[0] != 0);
    memBank = buffer[1];
    address = 0;
    address += (ushort)(((ushort)buffer[2]) << 8);
    address += (ushort)(((ushort)buffer[3]) << 0);
    size = buffer[4];
    password.resize(4);
    password[0] = buffer[5];
    password[1] = buffer[6];
    password[2] = buffer[7];
    password[3] = buffer[8];

    return true;
}

/// <summary>
/// Tries to set the post detect READ settings
/// </summary>
/// <param name="on">Should post detect READ be on?</param>
/// <param name="memBank">Target Memory Bank</param>
/// <param name="address">Target Address</param>
/// <param name="size">Target Size</param>
/// <param name="password">Password</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::setPostDetectReadParams(bool on, byte memBank, ushort address, byte size, std::vector<byte> password)
{
    std::vector<byte> buffer;
    buffer.resize(9);
    buffer[0] = (byte)((on)?1:0);
    buffer[1] = memBank;
    buffer[2] = (byte)(address >> 8);
    buffer[3] = (byte)(address >> 0);
    buffer[4] = size;
    buffer[5] = password[0];
    buffer[6] = password[1];
    buffer[7] = password[2];
    buffer[8] = password[3];

    return setParam(0x0006, buffer);
}


/// <summary>
/// Tries to get the used read block size
/// </summary>
/// <param name="wordCount">The count of words, that are read with one gen2 read operation</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::getReadBlockSize(byte& wordCount)
{
    wordCount = 0;

    std::vector<byte> buffer;
    if (!getParam(0x002C, buffer))
        return false;

    if (buffer.size() != 1)
        return false;

    wordCount = buffer[0];

    return true;
}

/// <summary>
/// Tries to set the used read block size
/// </summary>
/// <param name="wordCount">The count of words, that are read with one gen2 read operation</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::setReadBlockSize(byte wordCount)
{
    std::vector<byte> buffer;
    buffer.resize(1);
    buffer[0] = wordCount;

    return setParam(0x002C, buffer);
}

//************************************************************** Settings

//************************************************************** Tag-Functions

/// <summary>
/// Tries to execute a block-write command
/// </summary>
/// <param name="epc">The known EPC of the tag</param>
/// <param name="mem_bank">The retrieved handle</param>
/// <param name="address">The known EPC of the tag</param>
/// <param name="passwd">The retrieved handle</param>
/// <param name="blockSize">The block size of BlockWrite that should be used in words</param>
/// <param name="data">The retrieved handle</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::blockWriteToTag(std::vector<byte> epc, byte mem_bank, ushort address, std::vector<byte> passwd, byte blockSize, std::vector<byte> data)
{
    if (data.size() > 220)
    {
        Global::trc(3, "BlockWrite To Tag - NOK - Data");
        return false;
    }

    if (passwd.size() != 4)
    {
        Global::trc(3, "BlockWrite To Tag - NOK - Data");
        return false;
    }

    int index = 0;
    std::vector<byte> payload;
    payload.resize(6 + epc.size() + passwd.size() + data.size());

    payload[index++] = (byte)epc.size();
    Array::copy(epc, 0, payload, index, epc.size());
    index += epc.size();

    payload[index++] = (byte)mem_bank;
    payload[index++] = (byte)(address >> 8);
    payload[index++] = (byte)address;
    Array::copy(passwd, 0, payload, index, passwd.size());
    index += passwd.size();

    payload[index++] = (byte)blockSize;
    payload[index++] = (byte)data.size();
    Array::copy(data, 0, payload, index, data.size());
    index += data.size();

    std::vector<byte> resultData;

    if (!customTagCommand((byte)0x03, payload, resultData))
        return false;

    return true;
}

/// <summary>
/// Retrieves the handle of a tag
/// </summary>
/// <param name="epc">The known EPC of the tag</param>
/// <param name="handle">The retrieved handle</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::getTagHandle(std::vector<byte> epc, std::vector<byte>& handle)
{
    int index = 0;
    std::vector<byte> payload;
    payload.resize(1 + epc.size());

    payload[index++] = (byte)epc.size();
    Array::copy(epc, 0, payload, index, epc.size());
    index += epc.size();

    std::vector<byte> resultData;

    if (!customTagCommand((byte)0x04, payload, resultData))
        return false;

    if (resultData.size() < 2)
        return false;

    handle.resize(2);
    Array::copy(resultData, 0, handle, 0, 2);

    return true;
}

/// <summary>
/// Reads data direct from the handle of a tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="mem_bank">The mem bank to read from</param>
/// <param name="address">The mem address to read from</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="count">The count of data to read from</param>
/// <param name="data">The read data</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::readFromHandle(std::vector<byte> handle, byte mem_bank, ushort address, std::vector<byte> passwd, byte count, std::vector<byte>& data)
{
    if (handle.size() != 2)
    {
        Global::trc(3, "Read From Handle - NOK - Data");
        return false;
    }

    if (passwd.size() != 4)
    {
        Global::trc(3, "Read From Handle - NOK - Data");
        return false;
    }

    int index = 0;
    std::vector<byte> payload;
    payload.resize(4 + handle.size() + passwd.size());

    Array::copy(handle, 0, payload, index, handle.size());
    index += handle.size();

    payload[index++] = (byte)mem_bank;
    payload[index++] = (byte)(address >> 8);
    payload[index++] = (byte)address;

    Array::copy(passwd, 0, payload, index, passwd.size());
    index += passwd.size();

    payload[index++] = (byte)count;

    std::vector<byte> resultData;

    if (!customTagCommand((byte)0x05, payload, resultData))
        return false;

    data.resize(resultData.size() - 1);
    Array::copy(resultData, 1, data, 0, resultData.size() - 1);

    return true;
}

/// <summary>
/// Writes data direct to the handle of a tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="mem_bank">The mem bank to write to</param>
/// <param name="address">The mem address to write to</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="data">The data to be written</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::writeToHandle(std::vector<byte> handle, byte mem_bank, ushort address, std::vector<byte> passwd, std::vector<byte> data)
{
    if (handle.size() != 2)
    {
        Global::trc(3, "Write To Handle - NOK - Data");
        return false;
    }

    if (data.size() > 220)
    {
        Global::trc(3, "Write To Handle - NOK - Data");
        return false;
    }

    if (passwd.size() != 4)
    {
        Global::trc(3, "Write To Handle - NOK - Data");
        return false;
    }

    int index = 0;
    std::vector<byte> payload;
    payload.resize(4 + handle.size() + passwd.size() + data.size());

    Array::copy(handle, 0, payload, index, handle.size());
    index += handle.size();

    payload[index++] = (byte)mem_bank;
    payload[index++] = (byte)(address >> 8);
    payload[index++] = (byte)address;

    Array::copy(passwd, 0, payload, index, passwd.size());
    index += passwd.size();

    payload[index++] = (byte)data.size();
    Array::copy(data, 0, payload, index, data.size());
    index += data.size();

    std::vector<byte> resultData;

    if (!customTagCommand((byte)0x06, payload, resultData))
        return false;

    return true;
}

/// <summary>
/// BlockWrites data direct to the handle of a tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="mem_bank">The mem bank to write to</param>
/// <param name="address">The mem address to write to</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="blockSize">The block size of BlockWrite that should be used in words</param>
/// <param name="data">The data to be written</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::blockWriteToHandle(std::vector<byte> handle, byte mem_bank, ushort address, std::vector<byte> passwd, byte blockSize, std::vector<byte> data)
{
    if (handle.size() != 2)
    {
        Global::trc(3, "BlockWrite To Handle - NOK - Data");
        return false;
    }

    if (data.size() > 220)
    {
        Global::trc(3, "BlockWrite To Handle - NOK - Data");
        return false;
    }

    if (passwd.size() != 4)
    {
        Global::trc(3, "BlockWrite To Handle - NOK - Data");
        return false;
    }

    int index = 0;
    std::vector<byte> payload;
    payload.resize(5 + handle.size() + passwd.size() + data.size());

    Array::copy(handle, 0, payload, index, handle.size());
    index += handle.size();

    payload[index++] = (byte)mem_bank;
    payload[index++] = (byte)(address >> 8);
    payload[index++] = (byte)address;

    Array::copy(passwd, 0, payload, index, passwd.size());
    index += passwd.size();

    payload[index++] = (byte)blockSize;
    payload[index++] = (byte)data.size();
    Array::copy(data, 0, payload, index, data.size());
    index += data.size();


    std::vector<byte> resultData;

    if (!customTagCommand((byte)0x08, payload, resultData))
        return false;

    return true;
}


/// <summary>
/// Sends a custom tag command to the tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="command">The two byte command to be sent</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="txBitCount">The count of bits to be sent</param>
/// <param name="txBits">The bits to be sent</param>
/// <param name="headerBit">Flag to signalize, if header bit was received</param>
/// <param name="rxBytes">The bytes received from the tag</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::customGen2Command(std::vector<byte> handle, std::vector<byte> command, std::vector<byte> passwd, byte txBitCount, std::vector<byte> txBits, bool& headerBit, std::vector<byte>& rxBytes)
{
    std::vector<byte> additionalInfo;
    return customGen2Command(handle, command, passwd, txBitCount, txBits, additionalInfo, headerBit, rxBytes);
}

/// <summary>
/// Sends a custom tag command to the tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="command">The two byte command to be sent</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="txBitCount">The count of bits to be sent</param>
/// <param name="txBits">The bits to be sent</param>
/// <param name="headerBit">Flag to signalize, if header bit was received</param>
/// <param name="estimatedRxCount">The estimated count of rx bits</param>
/// <param name="rxBytes">The bytes received from the tag</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::customGen2Command(std::vector<byte> handle, std::vector<byte> command, std::vector<byte> passwd, byte txBitCount, std::vector<byte> txBits, byte estimatedRxCount, bool& headerBit, std::vector<byte>& rxBytes)
{
    std::vector<byte> additionalInfo;
    additionalInfo.push_back((byte)estimatedRxCount);
    return customGen2Command(handle, command, passwd, txBitCount, txBits, additionalInfo, headerBit, rxBytes);
}

/// <summary>
/// Sends a custom tag command to the tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="command">The two byte command to be sent</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="txBitCount">The count of bits to be sent</param>
/// <param name="txBits">The bits to be sent</param>
/// <param name="headerBit">Flag to signalize, if header bit was received</param>
/// <param name="estimatedRxCount">The estimated count of rx bits</param>
/// <param name="estimatedTagProcessingTime">The estimated prossing time in ms of the tag, before replying</param>
/// <param name="rxBytes">The bytes received from the tag</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::customGen2Command(std::vector<byte> handle, std::vector<byte> command, std::vector<byte> passwd, byte txBitCount, std::vector<byte> txBits, byte estimatedRxCount, byte estimatedTagProcessingTime, bool& headerBit, std::vector<byte>& rxBytes)
{
    std::vector<byte> additionalInfo;
    additionalInfo.push_back((byte)estimatedRxCount);
    additionalInfo.push_back((byte)estimatedTagProcessingTime);
    return customGen2Command(handle, command, passwd, txBitCount, txBits, additionalInfo, headerBit, rxBytes);
}

/// <summary>
/// Sends a custom tag command to the tag
/// </summary>
/// <param name="handle">The handle of the tag</param>
/// <param name="command">The two byte command to be sent</param>
/// <param name="passwd">The passwd to be used</param>
/// <param name="txBitCount">The count of bits to be sent</param>
/// <param name="txBits">The bits to be sent</param>
/// <param name="headerBit">Flag to signalize, if header bit was received</param>
/// <param name="additionalInfo">Additional Info like estimated RX count or tag processing time</param>
/// <param name="rxBytes">The bytes received from the tag</param>
/// <returns>Success of the operation</returns>
bool CPPrfePURprotocolHandler::customGen2Command(std::vector<byte> handle, std::vector<byte> command, std::vector<byte> passwd, byte txBitCount, std::vector<byte> txBits, std::vector<byte> additionalInfo, bool& headerBit, std::vector<byte>& rxBytes)
{
    rxBytes.clear();
    headerBit = true;

    if (handle.size() != 2)
    {
        Global::trc(3, "Custom Tag Command Handle - NOK - Data");
        return false;
    }

    if (command.size() != 2 || passwd.size() != 4)
    {
        Global::trc(3, "Custom Tag Command Handle - NOK - Data");
        return false;
    }

    if (passwd.size() != 4)
    {
        Global::trc(3, "Custom Tag Command Handle - NOK - Data");
        return false;
    }

    byte txByteCount = (byte)(txBitCount >> 3);
    if ((txBitCount % 8) > 0)
        txByteCount++;

    if (txByteCount != txBits.size())
    {
        Global::trc(3, "Custom Tag Command Handle - NOK - Data");
        return false;
    }


    int index = 0;
    std::vector<byte> payload;
    payload.resize(1 + handle.size() + command.size() + passwd.size() + txBits.size() + additionalInfo.size());

    Array::copy(handle, 0, payload, index, handle.size());
    index += handle.size();

    Array::copy(command, 0, payload, index, command.size());
    index += command.size();

    Array::copy(passwd, 0, payload, index, passwd.size());
    index += passwd.size();

    payload[index++] = (byte)txBitCount;

    Array::copy(txBits, 0, payload, index, txBits.size());
    index += txBits.size();

    Array::copy(additionalInfo, 0, payload, index, additionalInfo.size());
    index += additionalInfo.size();

    std::vector<byte> resultData;

    if (!customTagCommand((byte)0x07, payload, resultData))
        return false;

    headerBit = resultData[0] != 0;

    rxBytes.resize(resultData.size() - 2);
    Array::copy(resultData, 2, rxBytes, 0, resultData.size() - 2);

    return true;
}

//**************************************************************



void CPPrfePURprotocolHandler::notificationISR(const std::vector<byte>& payload)
{
    if(payload.size() < 1){
        CPPrfeProtocolHandler::notificationISR(payload);
        return;
    }

    switch (payload[0])
    {
    case 0x02:
        if (m_purEventListener != 0)
            m_purEventListener->InventoryRoundEndedHandler();
        break;

    default:
        CPPrfeProtocolHandler::notificationISR(payload);
        break;
    }
}
