/*
 * Copyright (c) 2008-2016, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "QtDevice.h"

#include <iostream>
#include <vector>

#include <CPPrfeReaderInterface.h>

using namespace std;

QtDevice::QtDevice(QIODevice *device)
{
    m_device = device;
    m_threadInitialized = false;
}

bool QtDevice::open()
{
    if(this->isRunning())
        return false;

    m_threadInitialized = false;
    this->start(QThread::HighestPriority);

    while(!m_threadInitialized){
        msleep(10);
    }

    if(this != QThread::currentThread ())
        m_device->moveToThread(this);

    return true;
}

bool QtDevice::close()
{
    if(!this->isRunning())
        return false;

    quit();
    while(!isFinished ())
        sleep(1);

    return true;
}

bool QtDevice::send(const std::vector<byte> &data)
{
    QByteArray ba;
    ba.resize(data.size());
    for(uint i = 0; i < data.size(); i++)
        ba[i] = data.at(i);
    return m_sendBuffer->write(ba) == data.size();
}

void QtDevice::readFromDevice()
{
    QByteArray data = m_device->readAll();

    std::vector<byte> vec;
    vec.resize(data.size());
    for(int i = 0; i < data.size(); i++)
        vec[i] = data.at(i);

    CPPrfeReaderInterface::Global::trc(9, ">> " + QString(data.toHex().data()).toStdString());

    raiseDataReadEvent(vec);
}

void QtDevice::sendToDevice()
{
    if(!m_device->isOpen()){
        CPPrfeReaderInterface::Global::trc(9, "Trying to write but device is not open!");
        return;
    }

    QByteArray data = m_sendBuffer->readAll();
    m_device->write(data);

    CPPrfeReaderInterface::Global::trc(9, "<< " + QString(data.toHex().data()).toStdString());
}

void QtDevice::run()
{
    m_sendBuffer = new QrfeFifo();
    m_sendBuffer->open(QIODevice::ReadWrite);

    // connect to the ready read signal of the device
    connect(m_sendBuffer, 		SIGNAL(readyRead()),
            this, 				  SLOT(sendToDevice()), Qt::DirectConnection);
    connect(m_device, 			SIGNAL(readyRead()),
            this, 				  SLOT(readFromDevice()), Qt::DirectConnection);

    CPPrfeReaderInterface::Global::trc(9, "Worker thread initialized");

	m_threadInitialized = true;

    exec();

    if(m_device->isOpen())
        m_device->close();

    if(m_sendBuffer && m_sendBuffer->isOpen())
        m_sendBuffer->close();

    delete m_sendBuffer;
    delete m_device;
}

