﻿/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using CSrfeReaderInterface.device;
using CSrfeReaderInterface.protocol;
using CSrfeReaderInterface.impl.device;
using CSrfeReaderInterface;

namespace CSrfeTest_ConsoleApplication
{
    class TestProgram
    {
        public enum ConnectionType
        {
            SERIAL,
            TCP
        }

        public enum ReaderType
        {
            STANDARD,
            PUR,
            AUR,
        }

        protected ConnectionType _connectionType;
        protected string _serialPortName;
        protected string _ipAddress;
        protected ushort _tcpPort;

        protected ReaderType _readerType;

        protected CSrfeProtocolHandler _ph = null;
        protected bool _init = false;
        protected int _testTraceLevel = 0;

        protected CSrfeProtocolHandler.HeartBeatHandler _onHeratBeat = null;
        protected CSrfeProtocolHandler.StateChangedHandler _onStateChanged = null;
        protected CSrfeProtocolHandler.GpioValuesChangedHandler _onGpioValuesChanged = null;
        protected CSrfeProtocolHandler.CyclicInventoryHandler _onCyclicInventory = null;
        protected CSrfeProtocolHandler.ErrorEventHandler _onErrorOccured = null;



        public TestProgram()
        {
            _ph = null;
            _init = false;
        }

        public bool init(IProtocolDeviceInterface device, ReaderType type)
        {
            // create trace and turn off 
            Global.m_tracer = new Trace.ConsoleTrace();
            Global.m_tracer.TraceLevel = _testTraceLevel;

            _readerType = type;

            SerialDevice serial = device as SerialDevice;
            if (serial != null)
            {
                _serialPortName = serial.PortName();
                _connectionType = ConnectionType.SERIAL;
            }

            TCPDevice tcp = device as TCPDevice;
            if (tcp != null)
            {
                _ipAddress = tcp.IpAddress();
                _tcpPort = (ushort)tcp.Port();
                _connectionType = ConnectionType.TCP;
            }

            _ph = createInstance(device, type);

            if (_ph == null)
                return false;

            _ph.setHeartBeat(false, 0);

            // connect events to the local event handler
            _ph.HeartBeat += _onHeratBeat = new CSrfeProtocolHandler.HeartBeatHandler(onHeratBeat);
            _ph.StateChanged += _onStateChanged = new CSrfeProtocolHandler.StateChangedHandler(onStateChanged);
            _ph.GpioValuesChanged += _onGpioValuesChanged = new CSrfeProtocolHandler.GpioValuesChangedHandler(onGpioValuesChanged);
            _ph.CyclicInventory += _onCyclicInventory = new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventory);
            _ph.ErrorEvent += _onErrorOccured = new CSrfeProtocolHandler.ErrorEventHandler(onErrorOccured);

            _init = true;

            return true;
        }

        public bool exec()
        {
            if (!_init)
                return false;

            Console.Clear();
            Console.WriteLine("");
            printTestTable();

            Console.Write("Please choose: > ");
            string idStr = Console.ReadLine();

            Console.WriteLine("");

            if (idStr == "q")
            {
                Console.WriteLine("Exit");
                return false;
            }

            int id = 0;

            try
            {
                id = Convert.ToInt32(idStr);
            }
            catch (Exception /*e*/)
            {
                Console.WriteLine("Input string is not a sequence of digits.");
                return true;
            }

            if (callTestRoutine(id) == false)
            {
                Console.WriteLine("Not yet implemented");
            }


            Console.WriteLine("\n\n\n");
            Console.WriteLine("Press enter to proceed...");
            Console.ReadLine();

            return true;
        }


        #region Test Methods

            #region Common

                public void test_ReaderInfo()
                {
                    bool ok = false;

                    // get reader id
                    uint readerId = 0;
                    ok = _ph.getReaderID(out readerId);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get ReaderID");

                    // get reader type
                    uint readerType = 0;
                    ok = _ph.getReaderType(out readerType);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get ReaderType");

                    // get hardware revision
                    uint hwRev = 0;
                    ok = _ph.getHardwareRevision(out hwRev);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get HardwareRevision");

                    // get software revision
                    uint swRev = 0;
                    ok = _ph.getSoftwareRevision(out swRev);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get SoftwareRevision");

                    // get bootloader revision
                    uint blRev = 0;
                    ok = _ph.getBootloaderRevision(out blRev);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get BootloaderRevision");

                    // get current system
                    Constants.eRFE_CURRENT_SYSTEM system = 0;
                    ok = _ph.getCurrentSystem(out system);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get CurrentSystem");

                    // get current state
                    Constants.eRFE_CURRENT_READER_STATE state = 0;
                    ok = _ph.getCurrentState(out state);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get CurrentState");

                    // get status register
                    ulong statusReg = 0;
                    ok = _ph.getStatusRegister(out statusReg);
                    if (!ok)
                        Console.WriteLine("ERROR: Could not get StatusRegister");

                    // print out results
                    Console.WriteLine("Reader Information:");
                    Console.WriteLine("\t -> ReaderID       = 0x{0:X08}", readerId);
                    Console.WriteLine("\t -> ReaderType     = 0x{0:X08}", readerType);
                    Console.WriteLine("\t -> HardwareRev    = 0x{0:X08}", hwRev);
                    Console.WriteLine("\t -> SoftwareRev    = 0x{0:X08}", swRev);
                    Console.WriteLine("\t -> BootloaderRev  = 0x{0:X08}", blRev);
                    Console.WriteLine("\t -> Current System = {0}", system.ToString());
                    Console.WriteLine("\t -> Current State  = {0}", state.ToString());
                    Console.WriteLine("\t -> StatusRegister = 0x{0:X016}", statusReg);
                }

                public void test_Attenuation()
                {
                    Console.WriteLine("Testing attenuation settings:");

                    ushort attMax, attCur, attTemp;

                    // get attenuation values
                    Console.WriteLine("\t -> 1, Reading attenuation settings:");
                    _ph.getAttenuation(out attMax, out attCur);
                    Console.WriteLine("\t\t Attenuation Max={0} Current={1}", attMax, attCur);
                    Console.WriteLine("");

                    // set attenuation to fix value 10
                    Console.WriteLine("\t -> 2, Setting attenuation settings:");
                    _ph.setAttenuation(10);
                    Console.WriteLine("\t\t Set Attenuation to 10");
                    Console.WriteLine("");

                    // get attenuation settings and check if the fix value is set
                    Console.WriteLine("\t -> 3, Reading attenuation settings:");
                    _ph.getAttenuation(out attMax, out attTemp);
                    Console.WriteLine("\t\t Attenuation Max={0} Current={1}", attMax, attTemp);
                    Console.WriteLine("\t\t Current attenuation {0} == 10?", attTemp);
                    if (attTemp != 10)
                    {
                        Console.WriteLine("ERROR------------------ Set Attenuation is not the Current Attenuation");
                        return;
                    }
                    Console.WriteLine("\t\t OK\n");

                    // retore attenuation to the previous value
                    Console.WriteLine("\t -> 4, Restore attenuation settings:");
                    _ph.setAttenuation(attCur);
                    Console.WriteLine("\t\t Set Attenuation to {0}", attCur);
                    Console.WriteLine("");

                    // check the set values again
                    Console.WriteLine("\t -> 5, Reading attenuation settings:");
                    _ph.getAttenuation(out attMax, out attCur);
                    Console.WriteLine("\t\t Attenuation Max={0} Current={1}", attMax, attCur);
                    Console.WriteLine("");
                }

                public void test_Frequency()
                {
                    Console.WriteLine("Testing frequency settings:");

                    byte mode, maxCount;
                    List<uint> frequ;
                    List<uint> tempFrequ;

                    // get current frequency table
                    Console.WriteLine("\t -> 1, Reading frequency settings:");
                    _ph.getFrequency(out mode, out maxCount, out frequ);
                    Console.WriteLine("\t\t FrequencyTable Mode={0} Max={1} Current={2}", mode, maxCount, frequ.Count());
                    for (int i = 0; i < frequ.Count(); i++)
                    {
                        Console.WriteLine("\t\t\t {0} = {1}", i, frequ[i]);
                    }
                    Console.WriteLine("");

                    // create new frequency table and set this
                    Console.WriteLine("\t -> 2, Setting frequency settings:");
                    List<uint> newFreq = new List<uint>(new uint[] { 855000, 856000, 857000 });
                    _ph.setFrequency(mode, newFreq);
                    Console.WriteLine("\t\t Set FrequencyTable to Mode={0}", mode);
                    for (int i = 0; i < newFreq.Count(); i++)
                    {
                        Console.WriteLine("\t\t\t {0} = {1}", i, newFreq[i]);
                    }
                    Console.WriteLine("");

                    // retrieve the table again and check if the retireved table is the same like the set
                    Console.WriteLine("\t -> 3, Reading frequency settings:");
                    _ph.getFrequency(out mode, out maxCount, out tempFrequ);
                    Console.WriteLine("\t\t FrequencyTable Mode={0} Max={1} Current={2}", mode, maxCount, tempFrequ.Count());
                    for (int i = 0; i < tempFrequ.Count(); i++)
                    {
                        Console.WriteLine("\t\t\t {0} = {1}", i, tempFrequ[i]);
                    }
                    Console.WriteLine("\t\t Last set Frequency Table == Read Frequency Table?");
                    if (!newFreq.TrueForAll(tempFrequ.Contains) || !tempFrequ.TrueForAll(newFreq.Contains))
                    {
                        Console.WriteLine("ERROR------------------ Previous Set Frequency Table is not the Current Frequency Table!!");
                        return;
                    }
                    Console.WriteLine("\t\t OK\n");

                    // restore the previous frequency table
                    Console.WriteLine("\t -> 4, Restoring frequency settings:");
                    _ph.setFrequency(mode, frequ);
                    Console.WriteLine("\t\t Set FrequencyTable to Mode={0}", mode);
                    for (int i = 0; i < frequ.Count(); i++)
                    {
                        Console.WriteLine("\t\t\t {0} = {1}", i, frequ[i]);
                    }
                    Console.WriteLine("");

                    // recheck the frequency table
                    Console.WriteLine("\t -> 5, Reading frequency settings:");
                    _ph.getFrequency(out mode, out maxCount, out tempFrequ);
                    Console.WriteLine("\t\t FrequencyTable Mode={0} Max={1} Current={2}", mode, maxCount, tempFrequ.Count());
                    for (int i = 0; i < tempFrequ.Count(); i++)
                    {
                        Console.WriteLine("\t\t\t {0} = {1}", i, tempFrequ[i]);
                    }
                    Console.WriteLine("");
                }

                public void test_Sensitivity()
                {
                    Console.WriteLine("Testing sensitivity settings:");

                    short maxSens, minSens, curSens, temSens, actSens;

                    // get current sensitivity
                    Console.WriteLine("\t -> 1, Reading sensitivity settings:");
                    _ph.getSensitivity(out maxSens, out minSens, out curSens);
                    Console.WriteLine("\t\t Sensitivity Max={0} Min={1} Current={2}", maxSens, minSens, curSens);
                    Console.WriteLine("");

                    // set minimum sensitivity and read out the actual set value
                    Console.WriteLine("\t -> 2, Setting sensitivity settings to MIN:");
                    _ph.setSensitivity(minSens, out actSens);
                    Console.WriteLine("\t\t Sensitivity Min={0} Actual={1}", minSens, actSens);
                    Console.WriteLine("");

                    // retrieve the current sensitivity and check if the current value is the same like the actual set one 
                    Console.WriteLine("\t -> 3, Reading sensitivity settings:");
                    _ph.getSensitivity(out maxSens, out minSens, out temSens);
                    Console.WriteLine("\t\t Sensitivity Max={0} Min={1} Current={2}", maxSens, minSens, temSens);
                    Console.WriteLine("\t\t Current Sensitivity {0} == {1} Actual Set Sensitivirty?", temSens, actSens);
                    if (actSens != temSens)
                    {
                        Console.WriteLine("ERROR------------------ Actual Set Sensitivity is not the Current Sensitivity");
                        return;
                    }
                    Console.WriteLine("\t\t OK\n");

                    // set maximum sensitivity and read out the actual set value
                    Console.WriteLine("\t -> 4, Setting sensitivity settings to MAX:");
                    _ph.setSensitivity(maxSens, out actSens);
                    Console.WriteLine("\t\t Sensitivity Max={0} Actual={1}", maxSens, actSens);
                    Console.WriteLine("");


                    // retrieve the current sensitivity and check if the current value is the same like the actual set one 
                    Console.WriteLine("\t -> 5, Reading sensitivity settings:");
                    _ph.getSensitivity(out maxSens, out minSens, out temSens);
                    Console.WriteLine("\t\t Sensitivity Max={0} Min={1} Current={2}", maxSens, minSens, temSens);
                    Console.WriteLine("\t\t Current Sensitivity {0} == {1} Actual Set Sensitivirty?", temSens, actSens);
                    if (actSens != temSens)
                    {
                        Console.WriteLine("ERROR------------------ Actual Set Sensitivity is not the Current Sensitivity");
                        return;
                    }
                    Console.WriteLine("\t\t OK\n");

                    // restore the previous set value
                    Console.WriteLine("\t -> 6, Restore sensitivity settings:");
                    _ph.setSensitivity(curSens, out actSens);
                    Console.WriteLine("\t\t Sensitivity Cur={0} Actual={1}", curSens, actSens);
                    Console.WriteLine("\t\t Current Sensitivity {0} == {1} Actual Set Sensitivirty?", curSens, actSens);
                    if (actSens != curSens)
                    {
                        Console.WriteLine("ERROR------------------ Actual Set Sensitivity is not the Current Sensitivity");
                        return;
                    }
                    Console.WriteLine("\t\t OK\n");
                }

                public void test_Heartbeat()
                {
                    Console.WriteLine("Testing Heartbeat:");

                    // turn on heartbeat with an interval of 250ms
                    Console.WriteLine("\t -> 1, Setting Heartbeat ON with interval 250ms (Press Enter to Stop)");
                    _lastHeartbeat = DateTime.Now;
                    _ph.setHeartBeat(true, 250);

                    // wait for <enter>
                    Console.ReadLine();

                    // turn off heartbeat
                    Console.WriteLine("\t -> 2, Setting Heartbeat OFF");
                    _ph.setHeartBeat(false, 250);
                }

                public void test_GPIO_Output()
                {
                    Console.WriteLine("Test GPIO - Output");

                    ulong mask, output, input;
                    if (!_ph.getGPIOCaps(out mask, out output, out input))
                    {
                        Console.WriteLine("ERROR: Could not get GPIO Caps: " + _ph.LastReturnCode.ToString());
                        return;
                    }

                    Console.WriteLine("Setting each pin output.");
                    if (!_ph.setGPIODirection(output))
                    {
                        Console.WriteLine("ERROR: Could not set GPIO direction: " + _ph.LastReturnCode.ToString());
                        return;
                    }

                    Console.WriteLine();
                    ulong test_mask = 0;
                    for (int i = 0; i < 32; i++)
                    {
                        test_mask = (ulong)(((int)1) << i);
                        if ((test_mask & output) == 0)
                            continue;

                        Console.WriteLine("Setting pin: 0x{0:X08}", test_mask);
                        if (!_ph.setGPIO(test_mask))
                        {
                            Console.WriteLine("ERROR: Could not set GPIO: " + _ph.LastReturnCode.ToString());
                            break;
                        }

                        Console.WriteLine("Press enter to clear the Pin and to proceed");
                        Console.ReadLine();

                        if (!_ph.clearGPIO(test_mask))
                        {
                            Console.WriteLine("ERROR: Could not clear GPIO: " + _ph.LastReturnCode.ToString());
                            break;
                        }
                    }

                    Console.WriteLine("Setting all Pins low.");
                    if (!_ph.clearGPIO(mask))
                    {
                        Console.WriteLine("ERROR: Could not clear GPIO: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                }

                public void test_GPIO_Input()
                {
                    Console.WriteLine("Test GPIO - Input");

                    ulong mask, output, input;
                    if (!_ph.getGPIOCaps(out mask, out output, out input))
                    {
                        Console.WriteLine("ERROR: Could not get GPIO Caps: " + _ph.LastReturnCode.ToString());
                        return;
                    }

                    Console.WriteLine("Setting each input pin as input: 0x{0:X08}", input);
                    if (!_ph.setGPIODirection(~input))
                    {
                        Console.WriteLine("ERROR: Could not set GPIO direction: " + _ph.LastReturnCode.ToString());
                        return;
                    }

                    Console.WriteLine();
                    Console.WriteLine("Press enter to clear the Pin and to proceed");
                    Console.ReadLine();


                    Console.WriteLine("Setting all pins output.");
                    if (!_ph.clearGPIO(output))
                    {
                        Console.WriteLine("ERROR: Could not clear GPIO: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                }

                public void test_AntennaSetting()
                {
                    Console.WriteLine("Trying Antenna Settings:");

                    // Check how much antennas the reader supports
                    Console.WriteLine("\t -> 1, Get antenna count:");
                    byte antennaCount = 0;
                    _ph.getAntennaCount(out antennaCount);
                    Console.WriteLine("\t\tThe reader has " + antennaCount + " antenna connectors.");
                    Console.WriteLine("");

                    if (antennaCount <= 1)
                    {
                        Console.WriteLine("\tThe reader only offers 1 antenna, so no other can be selected.");
                        return;
                    }

                    Console.WriteLine("\t -> 2, Select the antenne setup:");
                    Console.WriteLine("\t\t [0] - All antennas for 2 seconds");
                    for (byte i = 1; i <= antennaCount; i++)
                        Console.WriteLine("\t\t [{0}] - Antenna #{0} permanent", i);

                    // Get input
                    Console.Write("\t What antenna setup should be used? [0-{0}] (0)> ", antennaCount);
                    string aS = Console.ReadLine();

                    // Check input and parse value
                    byte antennaSetup = 0;
                    if (aS == "")
                        antennaSetup = 0;
                    else
                    {
                        try
                        {
                            antennaSetup = Convert.ToByte(aS);
                        }
                        catch (System.Exception)
                        {
                            Console.WriteLine("\t Could not convert the given string \"{0}\" to an integer.", aS);
                            return;
                        }
                    }

                    // Check if input is valid
                    if(antennaSetup > antennaCount)
                    {
                        Console.WriteLine("\t The selected antenna setup is not valid: \"{0}\"", antennaSetup);
                        return;
                    }

                    // Create sequence list
                    List<Tuple<byte, ulong>> sequence = new List<Tuple<byte,ulong>>();
                    if (antennaSetup == 0)
                    {
                        for (byte i = 0; i < antennaCount; i++)
                        {
                            sequence.Add(new Tuple<byte, ulong>((byte)(i+1), 2000));
                        }
                    }
                    else
                    {
                        sequence.Add(new Tuple<byte, ulong>(antennaSetup, 0xFFFFFFFF));
                    }

                    Console.WriteLine("");

                    Console.WriteLine("\t -> 3, Set the antenne setup:");

                    // Set the new created sequence
                    if (_ph.setAntennaSequence(sequence) != true)
                    {
                        Console.WriteLine("\t Failed to set the antenna sequence: " + _ph.LastReturnCode.ToString());
                        return;
                    }

                    Console.WriteLine("\t Set the configured antenna seuqence succefully.");
                }

                public void test_Inventory()
                {
                    Console.WriteLine("Testing Cyclic Inventory:");

                    // turn on cyclic inventory
                    Console.WriteLine("\t -> 1, Starting Cyclic Inventory (Press Enter to Stop)");
                    _tagReadCount = 0;
                    DateTime startTime = DateTime.Now;
                    _ph.setCyclicInventory(true);

                    // wait for <enter>
                    Console.ReadLine();

                    // turn off cyclic inventory and calculate read rate
                    _ph.setCyclicInventory(false);
                    double secs = DateTime.Now.Subtract(startTime).TotalSeconds;
                    Console.WriteLine("\t -> 2, Stopped Cyclic Inventry with a ReadRate of {0} reads/sec", _tagReadCount / secs);
                }

                public void test_SingleInventory()
                {
                    // do a single inventory and print out the list
                    Console.WriteLine("Do SingleInventory");
                    List<CSrfeProtocolHandler.TagEvent> tagList;

                    DateTime startTime = DateTime.Now;
                    _ph.doSingleInventory(out tagList);
                    DateTime stopTime = DateTime.Now;


                    Console.WriteLine("\t\t -> Found {0} Tags after {1}", tagList.Count, stopTime.Subtract(startTime).TotalMilliseconds);
                    int i = 1;
                    foreach (CSrfeProtocolHandler.TagEvent t in tagList)
                        Console.WriteLine("\t\t #{0} {1}", i++, t.ToString());
                }

                public void test_InventoryWithAntennaSwitch()
                {
                    byte antennaCount = 0;
                    bool res = _ph.getAntennaCount(out antennaCount);
                    if (antennaCount <= 1)
                    {
                        Console.WriteLine("Antenna Switching can't be tested with a 1 port reader");
                        return;
                    }

                    Console.WriteLine("Testing Cyclic Inventory with Antenna Switching:");

                    // get the set antenna sequence
                    List<Tuple<byte, ulong>> orig_sequence;
                    if (_ph.getAntennaSequence(out orig_sequence) != true)
                    {
                        Console.WriteLine("\t Failed to get the antenna sequence: " + _ph.LastReturnCode.ToString());
                        return;
                    }

                    // turn on cyclic inventory
                    Console.WriteLine("\t -> 1, Starting Cyclic Inventory (Press Enter to Proceed)");
                    _tagReadCount = 0;
                    DateTime startTime = DateTime.Now;
                    _ph.setCyclicInventory(true);

                    Console.WriteLine("\t -> 2, Scanning with the original antenna sequence (Press Enter to Proceed)");

                    for(byte i = 1; i <= antennaCount; i++)
                    {
                        // wait for <enter>
                        Console.ReadLine();

                        List<Tuple<byte, ulong>> sequence = new List<Tuple<byte,ulong>>();
                        sequence.Add(new Tuple<byte, ulong>(i, 0xFFFFFFFF));

                        Console.WriteLine("\t -> {0}, Activating only antenne #{1} (Press Enter to Proceed)", i + 2, i);

                        // Set the new created sequence
                        if (_ph.setAntennaSequence(sequence) != true)
                        {
                            Console.WriteLine("\t Failed to set the antenna sequence: " + _ph.LastReturnCode.ToString());
                            _ph.setCyclicInventory(false);
                            return;
                        }
                    }

                    // wait for <enter>
                    Console.ReadLine();

                    Console.WriteLine("\t -> {0}, Activating the original antenna sequence  (Press Enter to Proceed)", antennaCount + 2);

                    // Set the original sequence
                    if (_ph.setAntennaSequence(orig_sequence) != true)
                    {
                        Console.WriteLine("\t Failed to set the antenna sequence: " + _ph.LastReturnCode.ToString());
                        _ph.setCyclicInventory(false);
                        return;
                    }

                    // wait for <enter>
                    Console.ReadLine();

                    // turn off cyclic inventory and calculate read rate
                    _ph.setCyclicInventory(false);
                    double secs = DateTime.Now.Subtract(startTime).TotalSeconds;
                    Console.WriteLine("\t -> {0}, Stopped Cyclic Inventry with a ReadRate of {1} reads/sec", antennaCount+2, _tagReadCount / secs);
                }

            #endregion

            #region PUR

                public void test_InventoryRSSI()
                {
                    CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                    if (pur_ph == null)
                    {
                        Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                        return;
                    }

                    Console.WriteLine("Testing Cyclic Inventory with RSSI:");

                    // switch sending of rssi on for PUR reader
                    Console.WriteLine("\t -> 1, Switching RSSI On");
                    _ph.setParam(0x02, new byte[] { 0x01 });

                    // turn on cyclic inventory
                    Console.WriteLine("\t -> 2, Starting Cyclic Inventory (Press Enter to Stop)");
                    _tagReadCount = 0;
                    DateTime startTime = DateTime.Now;
                    _ph.setCyclicInventory(true);

                    // wait for <enter>
                    Console.ReadLine();

                    // turn off cyclic inventory and calculate read rate
                    _ph.setCyclicInventory(false);
                    double secs = DateTime.Now.Subtract(startTime).TotalSeconds;
                    Console.WriteLine("\t -> 3, Stopped Cyclic Inventry with a ReadRate of {0} reads/sec", _tagReadCount / secs);

                    // switch sending of rssi off
                    Console.WriteLine("\t -> 4, Switching RSSI On");
                    _ph.setParam(0x02, new byte[] { 0x00 });
                }

                public void test_InventoryPostDetectREAD()
                {
                    CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                    if (pur_ph == null)
                    {
                        Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                        return;
                    }

                    Console.WriteLine("Testing Cyclic Inventory with PostDetect READ:");

                    // switch PostDetect READ On
                    Console.WriteLine("\t -> 1, Switching PostDetect READ On");
                    byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                    pur_ph.setPostDetectReadParams(true, 2, 0, 6, passwd);

                    // turn on cyclic inventory
                    Console.WriteLine("\t -> 2, Starting Cyclic Inventory (Press Enter to Stop)");
                    _tagReadCount = 0;
                    DateTime startTime = DateTime.Now;
                    _ph.setCyclicInventory(true);

                    // wait for <enter>
                    Console.ReadLine();

                    // turn off cyclic inventory and calculate read rate
                    _ph.setCyclicInventory(false);
                    double secs = DateTime.Now.Subtract(startTime).TotalSeconds;
                    Console.WriteLine("\t -> 3, Stopped Cyclic Inventry with a ReadRate of {0} reads/sec", _tagReadCount / secs);

                    // switch PostDetect READ Off
                    Console.WriteLine("\t -> 4, Switching PostDetect READ Off");
                    pur_ph.setPostDetectReadParams(false, 2, 0, 6, passwd);
                }

                public void test_ReadTIDUM()
                {
                    CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                    if (pur_ph == null)
                    {
                        Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                        return;
                    }

                    if (!check_antennaSelection())
                        return;

                    Console.WriteLine("Trying to read tid register:");

                    // do a single inventory and print out the tag list with an index to select one tag
                    Console.WriteLine("\t -> 1, Searching for tags:");
                    List<CSrfeProtocolHandler.TagEvent> tagList;
                    _ph.doSingleInventory(out tagList);
                    Console.WriteLine("\t\t Found {0} Tags", tagList.Count);
                    if (tagList.Count() > 0)
                    {
                        for (int i = 0; i < tagList.Count(); i++)
                            Console.WriteLine("\t\t\t [{0}]: {1}", i + 1, BitConverter.ToString(tagList[i].tagId));
                    }
                    else
                    {
                        Console.WriteLine("\t\t Found 0 Tags, stopping the test...");
                        return;
                    }
                    Console.WriteLine("");

                    // read the selected index
                    Console.Write("\t -> 2, Select a tag by index: > ");
                    int k = Console.ReadLine()[0];
                    int index = k - '0' - 1;
                    if (index < 0 || index >= tagList.Count())
                    {
                        Console.WriteLine("\t\t Selected unkown tag, stopping the test...");
                        return;
                    }
                    Console.WriteLine("");

                    ////////////////////////////////////////////////// READ
                    // try to read the first 4 bytes of the TID membank of the selected tag
                    Console.WriteLine("\t -> 3, Trying to read TID register:");
                    byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                    byte[] data;
                    if (_ph.readFromTag(tagList[index].tagId, (byte)0x02, 0, passwd, 4, out data))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < data.Length; i++)
                        {
                            Console.Write("{0:X02}-", data[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed");
                    }
                    Console.WriteLine("");

                    ////////////////////////////////////////////////// READ
                    // try to read the whole user memory bank
                    Console.WriteLine("\t -> 4, Trying to read UM register:");
                    if (_ph.readFromTag(tagList[index].tagId, 0x03, 0, passwd, 0, out data))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < data.Length; i++)
                        {
                            Console.Write("{0:X02}-", data[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");
                }

                public void test_ReadWriteUser()
                {
                    CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                    if (pur_ph == null)
                    {
                        Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                        return;
                    }

                    if (!check_antennaSelection())
                        return;

                    Console.WriteLine("Trying to read and write user register:");

                    // do a single inventory and print out the tag list with an index to select one tag
                    Console.WriteLine("\t -> 1, Searching for tags:");
                    List<CSrfeProtocolHandler.TagEvent> tagList;
                    _ph.doSingleInventory(out tagList);
                    Console.WriteLine("\t\t Found {0} Tags", tagList.Count());
                    if (tagList.Count() > 0)
                    {
                        for (int i = 0; i < tagList.Count(); i++)
                            Console.WriteLine("\t\t\t [{0}]: {1}", i + 1, BitConverter.ToString(tagList[i].tagId));
                    }
                    else
                    {
                        Console.WriteLine("\t\t Found 0 Tags, stopping the test...");
                        return;
                    }
                    Console.WriteLine("");

                    // read the selected index
                    Console.Write("\t -> 2, Select a tag by index: > ");
                    int k = Console.ReadLine()[0];
                    int index = k - '0' - 1;
                    if (index < 0 || index >= tagList.Count())
                    {
                        Console.WriteLine("\t\t Selected unkown tag, stopping the test...");
                        return;
                    }
                    Console.WriteLine("");

                    byte memBank = 0x03;
                    byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                    byte[] userMemBefore;
                    byte[] userMemAfter;
                    byte[] old4Bytes = new byte[4];
                    byte[] new4Bytes = new byte[4] { 0x11, 0x22, 0x33, 0x44 };

                    // try to read the whole user memory bank
                    Console.WriteLine("\t -> 3, Trying to Read all bytes from user mem");
                    if (_ph.readFromTag(tagList[index].tagId, memBank, 0, passwd, 0, out userMemBefore))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < userMemBefore.Length; i++)
                        {
                            Console.Write("{0:X02}-", userMemBefore[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");

                    // save the old first 4 bytes of the user memory
                    old4Bytes[0] = userMemBefore[0];
                    old4Bytes[1] = userMemBefore[1];
                    old4Bytes[2] = userMemBefore[2];
                    old4Bytes[3] = userMemBefore[3];

                    // try to write 4 dummy bytes to the user mem
                    Console.WriteLine("\t -> 4, Trying to Wrtie data:");
                    Console.WriteLine("\t\t\t {0}", BitConverter.ToString(new4Bytes));
                    if (_ph.writeToTag(tagList[index].tagId, memBank, 0, passwd, new4Bytes))
                    {
                        Console.WriteLine("\t\t Wrtie succeeded");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Wrtie failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");

                    // try to read the whole user me again
                    Console.WriteLine("\t -> 5, Trying to Read written data");
                    if (_ph.readFromTag(tagList[index].tagId, memBank, 0, passwd, 0, out userMemAfter))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < userMemAfter.Length; i++)
                        {
                            Console.Write("{0:X02}-", userMemAfter[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");

                    // try to write the old bytes into the memory
                    Console.WriteLine("\t -> 6, Trying to restore the bytes");
                    if (_ph.writeToTag(tagList[index].tagId, memBank, 0, passwd, userMemBefore))
                    {
                        Console.WriteLine("\t\t Wrtie succeeded");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Wrtie failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");

                    // again read the restored bytes from the tag
                    Console.WriteLine("\t -> 7, Trying to Read written data");
                    if (_ph.readFromTag(tagList[index].tagId, memBank, 0, passwd, 4, out userMemAfter))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data: ");
                        Console.Write("\t\t\t {0}", BitConverter.ToString(userMemAfter));
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");
                }

                public void test_ReadBlockWriteUser()
                {
                    byte blockSize = 2;
                    byte wordCount = 16;

                    CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                    if (pur_ph == null)
                    {
                        Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                        return;
                    }

                    if (!check_antennaSelection())
                        return;

                    Console.WriteLine("Trying to read and write user register:");

                    Console.Write("What wordCount should be used? [1-110] (16)> ");
                    string wC = Console.ReadLine();

                    if (wC == "")
                        wordCount = 16;
                    else
                    {
                        try
                        {
                            wordCount = Convert.ToByte(wC);
                        }
                        catch (System.Exception)
                        {
                            Console.WriteLine("Could not convert the given string \"{0}\" to an integer.", wC);
                            return;
                        }
                    }

                    Console.Write("What blockSize should be used? [1-4] (2)> ");
                    string bS = Console.ReadLine();
                    if (bS == "")
                        blockSize = 2;
                    else
                    {
                        try
                        {
                            blockSize = Convert.ToByte(bS);
                        }
                        catch (System.Exception)
                        {
                            Console.WriteLine("Could not convert the given string \"{0}\" to an integer.", wC);
                            return;
                        }
                    }


                    int size = wordCount * 2;


                    // do a single inventory and print out the tag list with an index to select one tag
                    Console.WriteLine("\t -> 1, Searching for tags:");
                    List<CSrfeProtocolHandler.TagEvent> tagList;
                    _ph.doSingleInventory(out tagList);
                    Console.WriteLine("\t\t Found {0} Tags", tagList.Count());
                    if (tagList.Count() > 0)
                    {
                        for (int i = 0; i < tagList.Count(); i++)
                            Console.WriteLine("\t\t\t [{0}]: {1}", i + 1, BitConverter.ToString(tagList[i].tagId));
                    }
                    else
                    {
                        Console.WriteLine("\t\t Found 0 Tags, stopping the test...");
                        return;
                    }
                    Console.WriteLine("");

                    // read the selected index
                    Console.Write("\t -> 2, Select a tag by index: > ");
                    int k = Console.ReadLine()[0];
                    int index = k - '0' - 1;
                    if (index < 0 || index >= tagList.Count())
                    {
                        Console.WriteLine("\t\t Selected unkown tag, stopping the test...");
                        return;
                    }
                    Console.WriteLine("");

                    byte memBank = 0x03;
                    byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                    byte[] userMemBefore;
                    byte[] userMemAfter;
                    byte[] oldBytes = new byte[size];
                    byte[] newBytes = new byte[size];

                    for (int i = 0; i < size; i++)
                    {
                        newBytes[i] = (byte)(i + 1 + ((i + 1) * 16));
                    }

                    ////////////////////////////////////////////////// READ
                    // try to read the whole user memory bank
                    Console.WriteLine("\t -> 3, Trying to Read all bytes from user mem");
                    if (_ph.readFromTag(tagList[index].tagId, memBank, 0, passwd, 0, out userMemBefore))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < userMemBefore.Length; i++)
                        {
                            Console.Write("{0:X02}-", userMemBefore[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");

                    // save the old first 4 bytes of the user memory
                    for (int i = 0; i < size; i++)
                    {
                        oldBytes[i] = userMemBefore[i];
                    }


                    ////////////////////////////////////////////////// BLOCKWRITE
                    // try to write 4 dummy bytes to the user mem
                    Console.WriteLine("\t -> 4, Trying to Wrtie data:");
                    Console.WriteLine("\t\t\t {0}", BitConverter.ToString(newBytes));
                    if (pur_ph.blockWriteToTag(tagList[index].tagId, memBank, 0, passwd, blockSize, newBytes))
                    {
                        Console.WriteLine("\t\t Wrtie succeeded");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Wrtie failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");


                    ////////////////////////////////////////////////// READ
                    // try to read the whole user me again
                    Console.WriteLine("\t -> 5, Trying to Read written data");
                    if (_ph.readFromTag(tagList[index].tagId, memBank, 0, passwd, 0, out userMemAfter))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < userMemAfter.Length; i++)
                        {
                            Console.Write("{0:X02}-", userMemAfter[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");


                    ////////////////////////////////////////////////// BLOCKWRITE
                    // try to write the old bytes into the memory
                    Console.WriteLine("\t -> 6, Trying to restore the bytes");
                    if (pur_ph.blockWriteToTag(tagList[index].tagId, memBank, 0, passwd, blockSize, oldBytes))
                    {
                        Console.WriteLine("\t\t Wrtie succeeded");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Wrtie failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");


                    ////////////////////////////////////////////////// READ
                    // again read the restored bytes from the tag
                    Console.WriteLine("\t -> 7, Trying to Read written data");
                    if (_ph.readFromTag(tagList[index].tagId, memBank, 0, passwd, 0, out userMemAfter))
                    {
                        Console.WriteLine("\t\t Read succeeded, read data:");
                        Console.Write("\t\t\t ");
                        for (int i = 0; i < userMemAfter.Length; i++)
                        {
                            Console.Write("{0:X02}-", userMemAfter[i]);
                            if (((i + 1) % 8) == 0)
                                Console.Write("\n\t\t\t ");
                        }
                        Console.Write("\n");
                    }
                    else
                    {
                        Console.WriteLine("\t\t Read failed: " + _ph.LastReturnCode.ToString());
                        return;
                    }
                    Console.WriteLine("");
                }


                #region AN

                    public void test_AN001_ReadTIDFirstTag_Slow()
                    {
                        CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                        if (pur_ph == null)
                        {
                            Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                            return;
                        }

                        if (!check_antennaSelection())
                            return;

                        Console.WriteLine("Trying to read tid register:");

                        _storedTagIdSema = new Semaphore(0, 1000);
                        _storedTagEvent.Clear();
                        CSrfeProtocolHandler.CyclicInventoryHandler newEventHandler = new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventoryStoreTagId);
                        pur_ph.CyclicInventory += newEventHandler;

                        Console.WriteLine("\t -> 1, Searching for tags:");

                        _tagReadCount = 0;
                        pur_ph.setCyclicInventory(true);

                        _storedTagIdSema.WaitOne(5000);

                        pur_ph.setCyclicInventory(false);

                        if (_storedTagEvent.Count == 0)
                        {
                            Console.WriteLine("\t\t Did not find any tag...");
                            pur_ph.CyclicInventory -= newEventHandler;
                            return;
                        }

                        Console.WriteLine("\t\t Found Tag " + _storedTagEvent.FirstOrDefault().ToString());

                        // Set working antenna for read operation
                        if (_storedTagEvent.FirstOrDefault().hasAntenna)
                            _ph.setWorkingAntenna(_storedTagEvent.FirstOrDefault().antennaId);

                        // try to read the first 4 bytes of the TID membank of the detected tag
                        Console.WriteLine("\t -> 2, Trying to read first 4 bytes of TID register:");
                        byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                        byte[] data;

                        DateTime startTime = DateTime.Now;
                        bool result = pur_ph.readFromTag(_storedTagEvent.FirstOrDefault().tagId, (byte)0x02, 0, passwd, 2, out data);
                        DateTime stopTime = DateTime.Now;
                    
                        if (result)
                        {
                            Console.WriteLine("\t\t Read succeeded, read data: {0}", BitConverter.ToString(data));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Read failed");
                        }

                        Console.WriteLine("");
                        Console.WriteLine("Read Result {0}ms after the tag was detected", stopTime.Subtract(startTime).TotalMilliseconds);
                        Console.WriteLine("");

                        pur_ph.CyclicInventory -= newEventHandler;
                    }

                    public void test_AN001_ReadTIDFirstTag_Fast()
                    {
                        CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                        if (pur_ph == null)
                        {
                            Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                            return;
                        }

                        if (!check_antennaSelection())
                            return;

                        Console.WriteLine("Trying to read tid register:");

                        _storedTagIdSema = new Semaphore(0, 1);
                        _storedTagEvent.Clear();
                        CSrfeProtocolHandler.CyclicInventoryHandler newEventHandler = new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventoryStoreTagId);
                        pur_ph.CyclicInventory += newEventHandler;

                        Console.WriteLine("\t -> 1, Searching for tags:");

                        _tagReadCount = 0;
                        pur_ph.setCyclicInventory(true);

                        _storedTagIdSema.WaitOne(5000);
                        pur_ph.BlockCyclicInventoryInterrupts = true;

                        if (_storedTagEvent.Count == 0)
                        {
                            Console.WriteLine("\t\t Did not find any tag...");
                            pur_ph.setCyclicInventory(false);
                            pur_ph.CyclicInventory -= newEventHandler;
                            pur_ph.BlockCyclicInventoryInterrupts = false;
                            return;
                        }

                        Console.WriteLine("\t\t Found Tag " + _storedTagEvent.FirstOrDefault().ToString());

                        // Set working antenna for read operation
                        if (_storedTagEvent.FirstOrDefault().hasAntenna)
                            _ph.setWorkingAntenna(_storedTagEvent.FirstOrDefault().antennaId);

                        // try to read the first 4 bytes of the TID membank of the detected tag
                        Console.WriteLine("\t -> 2, Trying to read first 4 bytes of TID register:");
                        byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                        byte[] data;

                        DateTime startTime = DateTime.Now;
                        bool result = pur_ph.readFromTag(_storedTagEvent.FirstOrDefault().tagId, (byte)0x02, 0, passwd, 2, out data);
                        DateTime stopTime = DateTime.Now;

                        if (result)
                        {
                            Console.WriteLine("\t\t Read succeeded, read data: {0}", BitConverter.ToString(data));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Read failed");
                        }

                        Console.WriteLine("");
                        Console.WriteLine("Read Result {0}ms after the tag was detected", stopTime.Subtract(startTime).TotalMilliseconds);
                        Console.WriteLine("");

                        pur_ph.setCyclicInventory(false);
                        pur_ph.CyclicInventory -= newEventHandler;
                        pur_ph.BlockCyclicInventoryInterrupts = false;
                    }

                    public void test_AN001_ReadTIDFirstTag_Handle()
                    {
                        CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                        if (pur_ph == null)
                        {
                            Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                            return;
                        }

                        if (!check_antennaSelection())
                            return;

                        Console.WriteLine("Trying to read tid register from first tag:");

                        pur_ph.setParam(0x24, new byte[1] { 0x01 });        //send Handle ON
                        pur_ph.setParam(0x00, new byte[1] { 0x04 });        //inventory_secure_single_tag
                        pur_ph.setParam(0x03, new byte[1] { 0x02 });        //sendTagIDImmediatelyAndStop

                        _storedTagIdSema = new Semaphore(0, 1);
                        _storedTagEvent.Clear();
                        CSrfeProtocolHandler.CyclicInventoryHandler newEventHandler = new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventoryStoreTagId);
                        pur_ph.CyclicInventory += newEventHandler;

                        Console.WriteLine("\t -> 1, Searching for tags and reading from first");

                        pur_ph.setAntennaPower(true);

                        _tagReadCount = 0;
                        pur_ph.setCyclicInventory(true);

                        _storedTagIdSema.WaitOne(5000);
                        pur_ph.BlockCyclicInventoryInterrupts = true;

                        if (_storedTagEvent.Count == 0 || _storedTagEvent.FirstOrDefault().hasHandle == false)
                        {
                            Console.WriteLine("\t\t Did not find any tag...");
                            pur_ph.setCyclicInventory(false);
                            pur_ph.setAntennaPower(false);
                            pur_ph.setParam(0x24, new byte[1] { 0x00 });        //send Handle OFF
                            pur_ph.setParam(0x00, new byte[2] { 0x00, 0x02 });  //inventory_secure_multi_tag
                            pur_ph.setParam(0x03, new byte[1] { 0x00 });        //sendTagIDImmediately

                            pur_ph.CyclicInventory -= newEventHandler;
                            pur_ph.BlockCyclicInventoryInterrupts = false;
                            return;
                        }

                        Console.WriteLine("\t\t Found Tag " + _storedTagEvent.FirstOrDefault().ToString());

                        // Set working antenna for read operation
                        if (_storedTagEvent.FirstOrDefault().hasAntenna)
                            _ph.setWorkingAntenna(_storedTagEvent.FirstOrDefault().antennaId);

                        // try to read the first 4 bytes of the TID membank of the detected tag
                        Console.WriteLine("\t -> 2, Trying to read first 4 bytes of TID register:");
                        byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                        byte[] data;

                        DateTime startTime = DateTime.Now;
                        bool res = pur_ph.readFromHandle(_storedTagEvent.FirstOrDefault().handle, (byte)0x02, 0, passwd, 2, out data);
                        DateTime stopTime = DateTime.Now;

                        if (res)
                        {
                            Console.WriteLine("\t\t Read succeeded, read data: {0}", BitConverter.ToString(data));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Read failed");
                        }

                        Console.WriteLine("");
                        Console.WriteLine("Read Result {0}ms after the tag was detected", stopTime.Subtract(startTime).TotalMilliseconds);
                        Console.WriteLine("");

                        pur_ph.setCyclicInventory(false);
                        pur_ph.setAntennaPower(false);
                        pur_ph.setParam(0x24, new byte[1] { 0x00 });        //send Handle OFF
                        pur_ph.setParam(0x00, new byte[1] { 0x02 });        //inventory_secure_multi_tag
                        pur_ph.setParam(0x03, new byte[1] { 0x00 });        //sendTagIDImmediately
                        pur_ph.CyclicInventory -= newEventHandler;
                        pur_ph.BlockCyclicInventoryInterrupts = false;
                    }

                #endregion

                #region CUSTOM

                    public void test_SL900A_GetTemperature()
                    {
                        CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                        if (pur_ph == null)
                        {
                            Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                            return;
                        }

                        if (!check_antennaSelection())
                            return;

                        Console.WriteLine("Trying to send SL900A command:");

                        // do a single inventory and print out the tag list with an index to select one tag
                        Console.WriteLine("\t -> 1, Searching for tags:");
                        List<CSrfeProtocolHandler.TagEvent> tagList;
                        _ph.doSingleInventory(out tagList);
                        Console.WriteLine("\t\t Found {0} Tags", tagList.Count);
                        if (tagList.Count() > 0)
                        {
                            for (int i = 0; i < tagList.Count(); i++)
                                Console.WriteLine("\t\t\t [{0}]: {1}", i + 1, BitConverter.ToString(tagList[i].tagId));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Found 0 Tags, stopping the test...");
                            return;
                        }
                        Console.WriteLine("");

                        // read the selected index
                        Console.Write("\t -> 2, Select a tag by index: > ");
                        int k = Console.ReadLine()[0];
                        int index = k - '0' - 1;
                        if (index < 0 || index >= tagList.Count())
                        {
                            Console.WriteLine("\t\t Selected unkown tag, stopping the test...");
                            return;
                        }
                        Console.WriteLine("");

                        // try to read the first 4 bytes of the TID membank of the selected tag
                        Console.WriteLine("\t -> 3, Trying to execute command:");

                        pur_ph.setAntennaPower(true);

                        // get handle of tag
                        byte[] handle;
                        if (pur_ph.getTagHandle(tagList[index].tagId, out handle))
                        {
                            Console.WriteLine("\t\t Retrieved Handle: {0}", BitConverter.ToString(handle));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Retrieve handle failed:  + pur_ph.LastReturnCode.ToString()");
                            pur_ph.setAntennaPower(false);
                            return;
                        }

                        // password is zero
                        byte[] passwd = new byte[4] { 0, 0, 0, 0 };

                        // command to GetSensorValue is E0AD
                        byte[] command = new byte[2] { 0xE0, 0xAD };

                        // 8 bits should be sent (6bits 0 RFU, 2bits sensor type: Temperature sensor)
                        byte txBitCount = 8;
                        byte[] txBits = new byte[] { 0x00 };


                        bool headerBit = false;
                        byte[] rxBytes;

                        bool result = pur_ph.customGen2Command(handle, command, passwd, txBitCount, txBits, 49, out headerBit, out rxBytes);
                        string retCode = pur_ph.LastReturnCode.ToString();


                        pur_ph.setAntennaPower(false);

                        if (!result)
                        {
                            Console.WriteLine("\t\t Execute failed: " + retCode);
                            return;
                        }

                        Console.WriteLine("\t\t Execute succeeded");
                        Console.WriteLine("\t\t  Header Bit:    {0}", headerBit);
                        Console.WriteLine("\t\t  Received Data: {0}", BitConverter.ToString(rxBytes));

                        if (headerBit)
                        {
                            Console.WriteLine("\t\t Received header bit");
                            return;
                        }

                        // expect at least 2byte 
                        if (rxBytes.Count() < 2)
                        {
                            Console.WriteLine("\t\t Received too less bytes");
                            return;
                        }

                        ushort temp_data = 0;
                        temp_data = (ushort)(rxBytes[0] & 0x03);
                        temp_data <<= 8;
                        temp_data += (ushort)rxBytes[1];

                        // standard formula for standard config
                        double temp = (double)temp_data;
                        temp *= 0.18;
                        temp -= 89.3;

                        Console.WriteLine("\t\t  Temp:          {0} °C ({1})", temp, temp_data);

                        Console.WriteLine("");
                    }

                    public void test_4325_GetTemperature()
                    {
                        CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                        if (pur_ph == null)
                        {
                            Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                            return;
                        }

                        if (!check_antennaSelection())
                            return;

                        Console.WriteLine("Trying to send 4325 command:");

                        // do a single inventory and print out the tag list with an index to select one tag
                        Console.WriteLine("\t -> 1, Searching for tags:");
                        List<CSrfeProtocolHandler.TagEvent> tagList;
                        _ph.doSingleInventory(out tagList);
                        Console.WriteLine("\t\t Found {0} Tags", tagList.Count);
                        if (tagList.Count() > 0)
                        {
                            for (int i = 0; i < tagList.Count(); i++)
                                Console.WriteLine("\t\t\t [{0}]: {1}", i + 1, BitConverter.ToString(tagList[i].tagId));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Found 0 Tags, stopping the test...");
                            return;
                        }
                        Console.WriteLine("");

                        // read the selected index
                        Console.Write("\t -> 2, Select a tag by index: > ");
                        int k = Console.ReadLine()[0];
                        int index = k - '0' - 1;
                        if (index < 0 || index >= tagList.Count())
                        {
                            Console.WriteLine("\t\t Selected unkown tag, stopping the test...");
                            return;
                        }
                        Console.WriteLine("");

                        // try to read the first 4 bytes of the TID membank of the selected tag
                        Console.WriteLine("\t -> 3, Trying to execute command:");

                        pur_ph.setAntennaPower(true);

                        // get handle of tag
                        byte[] handle;
                        if (pur_ph.getTagHandle(tagList[index].tagId, out handle))
                        {
                            Console.WriteLine("\t\t Retrieved Handle: {0}", BitConverter.ToString(handle));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Retrieve handle failed: " + pur_ph.LastReturnCode.ToString());
                            pur_ph.setAntennaPower(false);
                            return;
                        }

                        // password is zero
                        byte[] passwd = new byte[4] { 0, 0, 0, 0 };

                        // command to GetSensorData is E001
                        byte[] command = new byte[2] { 0xE0, 0x01 };

                        // send 2 bits to the tag (Send UID: 0, New Sample: 1)
                        byte txBitCount = 2;
                        byte[] txBits = new byte[] { 0x40 };

                        bool headerBit = false;
                        byte[] rxBytes;

                        bool result = pur_ph.customGen2Command(handle, command, passwd, txBitCount, txBits, 97, out headerBit, out rxBytes);
                        string retCode = pur_ph.LastReturnCode.ToString();

                        pur_ph.setAntennaPower(false);
                        
                        if (!result)
                        {
                            Console.WriteLine("\t\t Execute failed: " + retCode);
                            pur_ph.setAntennaPower(false);
                            return;
                        }

                        Console.WriteLine("\t\t Execute succeeded");
                        Console.WriteLine("\t\t  Header Bit:    {0}", headerBit);
                        Console.WriteLine("\t\t  Received Data: {0}", BitConverter.ToString(rxBytes));

                        if (headerBit)
                        {
                            Console.WriteLine("\t\t Received header bit");
                            return;
                        }

                        // expect at least 4byte sensor data and 4byte utc time stamp
                        if (rxBytes.Count() < 8)
                        {
                            Console.WriteLine("\t\t Received too less bytes");
                            return;
                        }

                        // analyze the received sensor data 
                        short temp_data = 0;
                        if ((rxBytes[0] & 0x01) != 0)
                        {
                            temp_data |= 0xFF;
                            temp_data <<= 8;
                        }
                        temp_data |= (short)(0x0000 | rxBytes[1]);

                        // formula for temperature
                        double temp = temp_data * 0.25;

                        Console.WriteLine("\t\t  Temp:          {0} °C ({1})", temp, temp_data);

                        Console.WriteLine("");
                    }

                    public void test_DirectCommand()
                    {
                        CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                        if (pur_ph == null)
                        {
                            Console.WriteLine("Unable to test. Protocol implementation is not valid. Should be PUR.");
                            return;
                        }

                        if (!check_antennaSelection())
                            return;

                        Console.WriteLine("Trying to send SL900A command:");

                        // do a single inventory and print out the tag list with an index to select one tag
                        Console.WriteLine("\t -> 1, Searching for tags:");
                        List<CSrfeProtocolHandler.TagEvent> tagList;
                        _ph.doSingleInventory(out tagList);
                        Console.WriteLine("\t\t Found {0} Tags", tagList.Count);
                        if (tagList.Count() > 0)
                        {
                            for (int i = 0; i < tagList.Count(); i++)
                                Console.WriteLine("\t\t\t [{0}]: {1}", i + 1, BitConverter.ToString(tagList[i].tagId));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Found 0 Tags, stopping the test...");
                            return;
                        }
                        Console.WriteLine("");

                        // read the selected index
                        Console.Write("\t -> 2, Select a tag by index: > ");
                        int k = Console.ReadLine()[0];
                        int index = k - '0' - 1;
                        if (index < 0 || index >= tagList.Count())
                        {
                            Console.WriteLine("\t\t Selected unkown tag, stopping the test...");
                            return;
                        }
                        Console.WriteLine("");

                        // try to read the first 4 bytes of the TID membank of the selected tag
                        Console.WriteLine("\t -> 3, Trying to execute command:");

                        pur_ph.setAntennaPower(true);

                        // get handle of tag
                        byte[] handle;
                        if (pur_ph.getTagHandle(tagList[index].tagId, out handle))
                        {
                            Console.WriteLine("\t\t Retrieved Handle: {0}", BitConverter.ToString(handle));
                        }
                        else
                        {
                            Console.WriteLine("\t\t Retrieve handle failed:  + pur_ph.LastReturnCode.ToString()");
                            pur_ph.setAntennaPower(false);
                            return;
                        }

                        byte sendwCRC = 0x01;
                        byte sendwTRCal = 0x00;

                        // Read from TID 3 words at address 0
                        byte txBitCount = 42;
                        byte[] txBits = new byte[] { 0xC2, 0x80, 0x00, 0xC0, 0x00, 0x00 };
                        txBits[3] |= (byte)((handle[0] >> 2) & 0x3F);
                        txBits[4] |= (byte)((handle[0] << 6) & 0xC0);
                        txBits[4] |= (byte)((handle[1] >> 2) & 0x3F);
                        txBits[5] |= (byte)((handle[1] << 6) & 0xC0);


                        bool headerBit = false;
                        byte[] rxBytes;

                        bool result = pur_ph.directCommand(sendwCRC, sendwTRCal, txBitCount, txBits, (6 * 8) + 16, out headerBit, out rxBytes);
                        string retCode = pur_ph.LastReturnCode.ToString();

                        pur_ph.setAntennaPower(false);

                        if (!result)
                        {
                            Console.WriteLine("\t\t Execute failed: " + retCode);
                            return;
                        }

                        Console.WriteLine("\t\t Execute succeeded");
                        Console.WriteLine("\t\t  Header Bit:    {0}", headerBit);
                        Console.WriteLine("\t\t  Received Data: {0}", BitConverter.ToString(rxBytes));

                        if (headerBit)
                        {
                            Console.WriteLine("\t\t Received header bit");
                            return;
                        }

                        Console.WriteLine("");
                    }
                
                #endregion

            #endregion

        #endregion Test Methods





        #region Event Handler

        protected DateTime _lastHeartbeat = DateTime.Now;

            // event handler for the heartbeat event
            public void onHeratBeat(byte[] data)
            {
                Console.WriteLine("\t\t Heartbeat after {0} ms / data: {1}", DateTime.Now.Subtract(_lastHeartbeat).TotalMilliseconds, BitConverter.ToString(data));
                _lastHeartbeat = DateTime.Now;
            }

            // event handler for the state changed event
            public void onStateChanged(Constants.eRFE_CURRENT_READER_STATE newState)
            {
                Console.WriteLine("\t\t =============== State changed to: " + newState.ToString() + " ===============");
            }

            // event handler for the state changed event
            public void onGpioValuesChanged(ulong gpioValues)
            {
                Console.WriteLine("\t\t GPIO Values Changed: = 0x{0:X08}", gpioValues);
            }


            protected ulong _tagReadCount = 0;

            // event handler for cyclic inventory event
            public void onCyclicInventory(CSrfeProtocolHandler.TagEvent tagEvent)
            {
                Console.Write("\t\t{0:D8}  {1} ", ++_tagReadCount, BitConverter.ToString(tagEvent.tagId));

                if (tagEvent.hasAntenna)
                {
                    Console.Write("@ANT#{0}", tagEvent.antennaId);
                }

                if (tagEvent.hasMemory)
                {
                    Console.Write("MEM {0} @{1}.{2} ", BitConverter.ToString(tagEvent.memData), tagEvent.memBank, tagEvent.memAddr);
                }

                if (tagEvent.hasApplicationInfo)
                {
                    Console.Write("APP ");

                    ///////////////////////////////////////////////////////////////////////
                    // SMARFIT
                    // expect at least 2byte 
                    if (tagEvent.applicationInfo.Count() >= 2)
                    {
                        ushort temp_data = 0;
                        temp_data = (ushort)(tagEvent.applicationInfo[0] & 0x03);
                        temp_data <<= 8;
                        temp_data += (ushort)tagEvent.applicationInfo[1];

                        // standard formula for standard config
                        double temp = (double)temp_data;
                        temp *= 0.18;
                        temp -= 89.3;

                        Console.Write("Temp: {0} °C ({1})", temp, temp_data);
                    }
                    else
                    {
                        Console.Write("{0} ", BitConverter.ToString(tagEvent.applicationInfo));
                    }
                }

                Console.WriteLine("");

                if (tagEvent.hasRSSI)
                {
                    CSrfePURprotocolHandler pur_ph = _ph as CSrfePURprotocolHandler;
                    if (pur_ph != null)
                    {
                        Console.Write("\t\t\t\tQ: ");
                        for (int i = 0; i < tagEvent.rssi[0]; i++)
                            Console.Write("#");
                        Console.Write("\n");

                        Console.Write("\t\t\t\tI: ");
                        for (int i = 0; i < tagEvent.rssi[1]; i++)
                            Console.Write("#");
                        Console.Write("\n");
                    }

                    CSrfeAURprotocolHandler aur_ph = _ph as CSrfeAURprotocolHandler;
                    if (aur_ph != null)
                    {
                        short rssi = (short)((((ushort)tagEvent.rssi[0]) << 8) | ((ushort)tagEvent.rssi[1]));
                        Console.Write("\t\t\t\tRSSI: {0}\n", rssi);
                    }
                }
            }


            protected Semaphore _storedTagIdSema = null;
            protected byte[] _storedTagId = null;
            protected List<CSrfeProtocolHandler.TagEvent> _storedTagEvent = new List<CSrfeProtocolHandler.TagEvent>();
            
            // event handler for cyclic inventory event
            public void onCyclicInventoryStoreTagId(CSrfeProtocolHandler.TagEvent tagEvent)
            {
                if (_storedTagId == null)
                {
                    _storedTagId = new byte[tagEvent.tagId.Length];
                    tagEvent.tagId.CopyTo(_storedTagId, 0);
                }

                _storedTagEvent.Add(new CSrfeProtocolHandler.TagEvent(tagEvent));

                if(_storedTagIdSema != null )
                    _storedTagIdSema.Release();
            }

            public void onErrorOccured(byte[] data)
            {
                if (data.Length < 1)
                    return;

                byte errCode = data[0];
                byte[] payload = new byte[data.Length - 1];
                Array.Copy(data, 1, payload, 0, payload.Length);
                Constants.eRFE_ERR_CODE code = (Constants.eRFE_ERR_CODE)errCode;
                Console.Write("ERROR: " + code.ToString() + " -> " + BitConverter.ToString(payload) + "\n\r");
            }

        #endregion Event Handler


        #region Helper
            protected bool check_antennaSelection()
            {
                byte antennaCount = 0;
                if (!_ph.getAntennaCount(out antennaCount))
                {
                    Console.WriteLine("ERROR: Could not get the antenna count.");
                    return false;
                }
                if (antennaCount <= 1)
                    return true;

                List<Tuple<byte, ulong>> sequence;
                if (!_ph.getAntennaSequence(out sequence))
                {
                    Console.WriteLine("ERROR: Could not get the antenna sequence.");
                    return false;
                }

                if (sequence.Count() == 1)
                    return true;

                // Get input
                Console.Write("Which antenna should be used? [1-{0}] (1)> ", antennaCount);
                string aS = Console.ReadLine();

                // Check input and parse value
                byte antennaSetup = 1;
                if (aS == "")
                    antennaSetup = 1;
                else
                {
                    try
                    {
                        antennaSetup = Convert.ToByte(aS);
                    }
                    catch (System.Exception)
                    {
                        Console.WriteLine("\t Could not convert the given string \"{0}\" to an integer.", aS);
                        return false;
                    }
                }

                if (!_ph.setWorkingAntenna(antennaSetup))
                {
                    Console.WriteLine("ERROR: Could not set working antenna.");
                    return false;
                }

                return true;
            }

            private CSrfeProtocolHandler createInstance(IProtocolDeviceInterface device, ReaderType type)
            {
                CSrfeProtocolHandler p = createInstanceExtended(device, type);

                if (p != null)
                    return p;

                switch (type)
                {
                    case ReaderType.STANDARD:
                        return new CSrfeProtocolHandler(device);
                    case ReaderType.PUR:
                        return new CSrfePURprotocolHandler(device);
                    case ReaderType.AUR:
                        return new CSrfeAURprotocolHandler(device);
                }

                return null;
            }

            private void printTestTable()
            {
                Console.WriteLine("+------------------------ MENU ----------------------------+");
                Console.WriteLine("|    q = Quit                                              |");
                Console.WriteLine("|    1 = Query Reader Information                          |");
                Console.WriteLine("|    2 = Test Attenuation Settings                         |");
                Console.WriteLine("|    3 = Test Frequency Settings                           |");
                Console.WriteLine("|    4 = Test Sensitivity Settings                         |");
                Console.WriteLine("|    5 = Test Heartbeat                                    |");
                Console.WriteLine("|    6 = Test GPIO - Output                                |");
                Console.WriteLine("|    7 = Test GPIO - Input                                 |");
                Console.WriteLine("|    8 = Select antenna                                    |");
                Console.WriteLine("|   10 = Do SinglInventory                                 |");
                Console.WriteLine("|   11 = Start Inventory                                   |");
                Console.WriteLine("|   12 = Start Inventory with Antenna Switching            |");
                Console.WriteLine("+-PUR-----------------------------------------------------+");
                Console.WriteLine("|   21 = Start RSSI Inventory                              |");
                Console.WriteLine("|   22 = Start PostDetect READ Inventory                   |");
                Console.WriteLine("|   31 = Try to Read TID & UM                              |");
                Console.WriteLine("|   32 = Try to Read/Write User Mem                        |");
                Console.WriteLine("|   33 = Try to Read/Block-Write User Mem                  |");
                Console.WriteLine("+-AN-PUR---------------------------------------------------+");
                Console.WriteLine("|   41 = AN001 - PUR - Test Read TID of First Tag - Slow   |");
                Console.WriteLine("|   42 = AN001 - PUR - Test Read TID of First Tag - Fast   |");
                Console.WriteLine("|   43 = AN001 - PUR - Test Read TID of First Tag - Handle |");
                Console.WriteLine("+-CUSTOM-TAG-----------------------------------------------+");
                Console.WriteLine("|   51 = SL900A - Get Temperature                          |");
                Console.WriteLine("|   52 = 4325 - Get Temperature                            |");
                Console.WriteLine("|   53 = Direct Command                                    |");
                Console.WriteLine("+----------------------------------------------------------+");
                printTestTableExtended();
            }

            private bool callTestRoutine(int id)
            {
                if (id == 1)
                    test_ReaderInfo();
                else if (id == 2)
                    test_Attenuation();
                else if (id == 3)
                    test_Frequency();
                else if (id == 4)
                    test_Sensitivity();
                else if (id == 5)
                    test_Heartbeat();
                else if (id == 6)
                    test_GPIO_Output();
                else if (id == 7)
                    test_GPIO_Input();
                else if (id == 8)
                    test_AntennaSetting();
                else if (id == 10)
                    test_SingleInventory();
                else if (id == 11)
                    test_Inventory();
                else if (id == 12)
                    test_InventoryWithAntennaSwitch();

                else if (id == 21)
                    test_InventoryRSSI();
                else if (id == 22)
                    test_InventoryPostDetectREAD();

                else if (id == 31)
                    test_ReadTIDUM();
                else if (id == 32)
                    test_ReadWriteUser();
                else if (id == 33)
                    test_ReadBlockWriteUser();

                else if (id == 41)
                    test_AN001_ReadTIDFirstTag_Slow();
                else if (id == 42)
                    test_AN001_ReadTIDFirstTag_Fast();
                else if (id == 43)
                    test_AN001_ReadTIDFirstTag_Handle();

                else if (id == 51)
                    test_SL900A_GetTemperature();
                else if (id == 52)
                    test_4325_GetTemperature();
                else if (id == 53)
                    test_DirectCommand();

                else
                    return callTestRoutinesExtended(id);

                return true;
            }

        #endregion


        #region Overridables

            public virtual bool getTestConnection(out IProtocolDeviceInterface dev, out ReaderType type)
            {
                dev = null;
                type = ReaderType.STANDARD;
                return false;
            }

            protected virtual CSrfeProtocolHandler createInstanceExtended(IProtocolDeviceInterface device, ReaderType type)
            {
                return null;
            }

            protected virtual void printTestTableExtended()
            {
            }

            protected virtual bool callTestRoutinesExtended(int id)
            {
                return false;
            }
        
        #endregion

    }
}
