﻿/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using CSrfeReaderInterface.device;
using CSrfeReaderInterface.impl.device;
using CSrfeReaderInterface.protocol;
using System;
using System.Collections.Generic;
using System.Threading.Tasks;

namespace CSrfeTest_ReadSerialNumberTest
{
    class CSrfeTest_ReadSerialNumberTest
    {
        /**************************************************************************/
        // TODO: Change this for your device:
        private string SerialPort = "COM7";

        private ushort TIDSerialNr_WordAddress = 3;
        private byte   TIDSerialNr_ByteSize = 4;
        /**************************************************************************/

        protected CSrfePURprotocolHandler _ph = null;
        protected ulong _tagReadCount = 0;

        /// <summary>
        /// Empty constructor
        /// </summary>
        public CSrfeTest_ReadSerialNumberTest()
        {
        }

        /// <summary>
        /// Test routine
        /// </summary>
        public void execute()
        {
            bool ok = false;

            // Create communication device
            IProtocolDeviceInterface device = new SerialDevice(SerialPort);
            if (!device.Open())
            {
                Console.WriteLine("ERROR: Could not open device at " + SerialPort + " - Please change the com port in the code:");
                Console.WriteLine("  private string SerialPort = \"COMXX\";");

                // wait for <enter>
                Console.ReadLine();
                return;
            }

            // Create Protocol handler
            _ph = new CSrfePURprotocolHandler(device);
            _ph.CyclicInventory += new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventory);


            //***********************************************************************
            // Test communication by reading the reader id
            Console.WriteLine("\t -> 1, Read ReaderID");
            
            // get reader id
            uint readerId = 0;
            ok = _ph.getReaderID(out readerId);
            if (!ok)
            {
                Console.WriteLine("ERROR: Could not get ReaderID");
                return;
            }

            //***********************************************************************
            // Test the scan
            Console.WriteLine("Testing Cyclic Inventory:");

            // turn on cyclic inventory
            Console.WriteLine("\t -> 2, Starting Cyclic Inventory (Press Enter to Restart; Enter q to quit)");
            _tagReadCount = 0;
            DateTime startTime = DateTime.Now;


            string str = "";

            while (!str.Contains("q"))
            {
                // Stop on tag detect
                _ph.setParam(0x0003, new byte[] { 0x02 });

                // EPC size = 0 -> dynamic
                _ph.setParam(0x0023, new byte[] { 0x00 });

                // Read TID serial number of tag after detect
                byte[] passwd = new byte[4] { 0, 0, 0, 0 };
                _ph.setPostDetectReadParams(true, 0x02, TIDSerialNr_WordAddress, TIDSerialNr_ByteSize, passwd);

                _ph.setCyclicInventory(true, 30000);

                // wait for <enter>
                str = Console.ReadLine();
            }

            // turn off cyclic inventory and calculate read rate
            _ph.setCyclicInventory(false);
            double secs = DateTime.Now.Subtract(startTime).TotalSeconds;
            Console.WriteLine("\t -> 3, Stopped Cyclic Inventry with a ReadRate of {0} reads/sec", _tagReadCount / secs);

            // wait for <enter>
            Console.ReadLine();
        }

        /// <summary>
        /// Event handler for cyclic inventory event
        /// </summary>
        /// <param name="tagEvent">The tag event detected and sent by the reader</param>
        public void onCyclicInventory(CSrfeProtocolHandler.TagEvent tagEvent)
        {
            Task.Factory.StartNew(() => { handleTag(tagEvent); });
        }

        public void handleTag(CSrfeProtocolHandler.TagEvent tagEvent)
        {
            byte[] serialNr = null;
            if (tagEvent.hasMemory && tagEvent.memBank == 0x02 && tagEvent.memAddr == 3)
            {
                serialNr = tagEvent.memData;
            }
            if (serialNr == null)
            {
                if (!_ph.readFromTag(tagEvent.tagId, 0x02, TIDSerialNr_WordAddress, new byte[4] { 0, 0, 0, 0 }, TIDSerialNr_ByteSize, out serialNr))
                    serialNr = new byte[0];
            }
            Console.Write("\t\t{0:D8}  {1}  -  {2}\n", ++_tagReadCount, BitConverter.ToString(tagEvent.tagId), BitConverter.ToString(serialNr));
        }


        

        /// <summary>
        /// Main routine
        /// </summary>
        /// <param name="args">arguments</param>
        static void Main(string[] args)
        {
            CSrfeTest_ReadSerialNumberTest p = new CSrfeTest_ReadSerialNumberTest();
            p.execute();
        }
    }
}
