﻿/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using CSrfeReaderInterface;
using CSrfeReaderInterface.device;
using CSrfeReaderInterface.impl.device;
using CSrfeReaderInterface.protocol;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Text.RegularExpressions;
using System.Threading;

namespace CSrfeTest_CmdLineScanner
{
    class CSrfeTest_CmdLineScanner
    {
        private class ReadData
        {
            public int AntennaID;
            public ushort Attenuation;
            public ushort Frequency;
            public ushort LinkFrequency;
            public ushort Coding;
            public int ReadCount;

            public override string ToString()
            {
                string str = "";
                str += "[";
                str += "@" + string.Format("{0,2}", this.AntennaID);
                str += ":" + string.Format("{0,2}steps", this.Attenuation);
                str += "," + string.Format("{0}", ToString_Frequency(this.Frequency));
                str += "," + string.Format("{0}", ToString_LinkFrequency(this.LinkFrequency));
                str += "," + string.Format("{0}", ToString_Coding(this.Coding));
                str += " -> " + string.Format("{0,3}", this.ReadCount);
                str += "]";
                return str;
            }

            public string ToStringTech()
            {
                string str = "";
                str += "[";
                str += string.Format("{0}", this.AntennaID);
                str += ",";
                str += string.Format("{0}", this.Attenuation);
                str += ",";
                str += string.Format("{0}", this.Frequency);
                str += ",";
                str += string.Format("{0}", this.LinkFrequency);
                str += ",";
                str += string.Format("{0}", this.Coding);
                str += ",";
                str += string.Format("{0}", this.ReadCount);
                str += "]";

                return str;
            }
        };

        private bool _UseTCP = true;
        private string _IP = "127.0.0.1";
        private ushort _Port = 52470;
        private string _TTY = "COM1";

        private int _EstimatedCount = 4;
        private int _InvRounds = 1;
        private bool _ResetReader = false;
        private bool _PrintHelp = false;

        private CSrfePURprotocolHandler _ProtocolHandler = null;
        private int _ScanTimeOut;

        private bool _FirstStart = true;

        private int _CurrentAntennaSequenceIndex = 0;
        private List<ushort> _AntennaSequence = new List<ushort>();

        private int _CurrentAttenuationIndex = 0;
        private List<ushort> _Attenuation = new List<ushort>();

        private int _CurrentFrequencyIndex = 0;
        private List<ushort> _Frequency = new List<ushort>();

        private int _CurrentLfCodingIndex = 0;
        private List<Tuple<ushort, ushort>> _LfCoding = new List<Tuple<ushort, ushort>>();

        private int _PrintReadInfoLevel;

        private string _OutputFile;

        private Dictionary<byte[], List<ReadData>> _ResultData = new Dictionary<byte[], List<ReadData>>(new ByteArrayComparer());

        List<uint> _EtsiLowerFrequencies = new List<uint>();
        List<uint> _EtsiUpperFrequencies = new List<uint>();

        bool _Exit = false;


        /// <summary>
        /// Constructor
        /// </summary>
        public CSrfeTest_CmdLineScanner()
        {
            _ProtocolHandler = null;

            _CurrentAntennaSequenceIndex = 0;
            _CurrentAttenuationIndex = 0;
            _CurrentFrequencyIndex = 0;
            _CurrentLfCodingIndex = 0;

            // Setup frequency lists
            // Setup lower ETSI frequencies
            _EtsiLowerFrequencies.Add(865700);
            _EtsiLowerFrequencies.Add(866300);
            _EtsiLowerFrequencies.Add(866900);
            _EtsiLowerFrequencies.Add(867500);
            // Setup upper ETSI frequencies
            _EtsiUpperFrequencies.Add(916300);
            _EtsiUpperFrequencies.Add(917500);
            _EtsiUpperFrequencies.Add(918700);
            _EtsiUpperFrequencies.Add(919900);

            // Setup default values for members
            _ScanTimeOut = 500;
            _AntennaSequence.Clear();
            _AntennaSequence.Add(1);
            _AntennaSequence.Add(2);
            _AntennaSequence.Add(3);
            _AntennaSequence.Add(4);
            _Attenuation.Clear();
            _Attenuation.Add(0);
            _Frequency.Clear();
            _Frequency.Add(0);
            _Frequency.Add(1);
            _PrintReadInfoLevel = 1;
            _OutputFile = "";
        }

        /// <summary>
        /// Initialize the scan with the given arguments
        /// </summary>
        /// <param name="args">Arguments of the command line</param>
        /// <returns>Returns if the initialisation was successful</returns>
        public bool Init(string[] args)
        {
            // create trace and turn off 
            Global.m_tracer = new CSrfeTest_ConsoleApplication.Trace.ConsoleTrace();
            Global.m_tracer.TraceLevel = 0;

            Global.trc(3, "  -- init");

            if (!ParseArguments(args))
                return false;

            if(_PrintHelp)
            {
                PrintHelp();
                return false;
            }

            // Create communication device
            IProtocolDeviceInterface device = null;
            if (_UseTCP)
            {
                IPAddress ipAddress = IPAddress.Parse("127.0.0.1");
                if (_IP != "")
                    IPAddress.Parse(_IP);

                device = new TCPDevice(ipAddress, _Port);
                if (!device.Open())
                {
                    Console.WriteLine("ERROR: Could not connect to IP " + _IP + ":" + _Port + ".");
                    return false;
                }
            }
            else
            {
                device = new SerialDevice(_TTY);
                if (!device.Open())
                {
                    Console.WriteLine("ERROR: Could not open " + _TTY + ".");
                    return false;
                }
            }
            

            // Create Protocol handler
            _ProtocolHandler = new CSrfePURprotocolHandler(device);
            _ProtocolHandler.HeartBeat += new CSrfeProtocolHandler.HeartBeatHandler(onHeartBeat);
            _ProtocolHandler.CyclicInventory += new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventory);

            _FirstStart = true;
            return InitReader();
        }

        /// <summary>
        /// Main Function to handle the reader and start scanning
        /// </summary>
        /// <returns>The return code for Main</returns>
        public int Execute()
        {
            while(true)
            {
                bool res = RestartCyclicInventory();

                if (_Exit)
                    break;

                if (!res)
                {
                    Global.trc(3, "  -- exit");
                    Global.trc(1, "Exiting application");

                    Console.WriteLine("Reboot");
                    return -1;
                }

                Thread.Sleep(_ScanTimeOut);
            }

            Global.trc(3, "  -- exit");
            Global.trc(1, "Exiting application");

            _ProtocolHandler.setCyclicInventory(false);

            return 0;
        }

        /// <summary>
        /// Generates the output for the console and to the output file.
        /// </summary>
        private void PrintOutput()
        {
            Global.trc(3, "  -- printOutputAndExit");
            Global.trc(2, "Print results for count of tags: " + _ResultData.Keys.Count);

            string header = _ResultData.Keys.Count.ToString() + " Tags found";
            Console.WriteLine(header);

            if (_OutputFile.Length > 0)
                File.WriteAllText(_OutputFile, header);

            // Print the result
            foreach (byte[] tagId in _ResultData.Keys)
            {
                List<ReadData> list;
                _ResultData.TryGetValue(tagId, out list);
                Global.trc(2, " Print results for count of params: " + list.Count.ToString());

                string output = "";
                output += "EPC: " + BitConverter.ToString(tagId);

                string info_1 = " ";
                string info_2 = " ";
                string info_3 = " ";
                string info_tech = " ";

                {
                    int max = 0;
                    int min = 99999999;
                    int pos_max = 0;
                    int pos_min = 0;
                    for (int j = 0; j < list.Count; j++)
                    {
                        ReadData p = list[j];
                        if (p.ReadCount > max)
                        {
                            max = p.ReadCount;
                            pos_max = j;
                        }
                        if (p.ReadCount < min)
                        {
                            min = p.ReadCount;
                            pos_min = j;
                        }
                    }

                    ReadData params_max = list[pos_max];
                    ReadData params_min = list[pos_min];

                    info_1 = "MAX: " + params_max.ToString();

                    info_2 = info_1;
                    if (pos_max != pos_min)
                        info_2 += "  MIN: " + params_min.ToString();
                }

                {
                    string[] paramStr = new string[list.Count];
                    string[] paramStr_tech = new string[list.Count];
                    for (int j = 0; j < list.Count; j++)
                    {
                        ReadData p = list[j];
                        paramStr[j] = p.ToString();
                        paramStr_tech[j] = p.ToStringTech();
                    }

                    info_3 += string.Join(", ", paramStr);
                    info_tech += string.Join(";", paramStr_tech);
                }

                // Print to stdout
                {
                    string str = " ";
                    if (_PrintReadInfoLevel == 1)
                        str = info_1;
                    if (_PrintReadInfoLevel == 2)
                        str = info_2;
                    if (_PrintReadInfoLevel == 3)
                        str = info_3;
                    //QrfeReaderInterface::Global::trc(0, "  " + output + " " + str);
                    Console.WriteLine("  " + output + " - " + str);
                }

                // Print to file
                if (_OutputFile.Length > 0)
                {
                    using (StreamWriter sw = File.AppendText(_OutputFile))
                    {
                        sw.Write(output);
                        sw.Write(" ");
                        sw.Write(info_tech);
                        sw.Write("\r\n");
                    }
                }


            }

            // Exit
            _Exit = true;
        }

        /// <summary>
        /// Generates the help output.
        /// </summary>
        private void PrintHelp()
        {
            Console.WriteLine("");
            Console.WriteLine("");
            Console.WriteLine(" RF-Embedded RFID Command Line Scanner");
            Console.WriteLine(" =====================================");
            Console.WriteLine("");
            Console.WriteLine(" Parameters:");
            Console.WriteLine("  -ip         Specifies the ip and port of the reader");
            Console.WriteLine("                Example: -ip:192.168.0.100:52460");
            Console.WriteLine("                -> Reader is at IP 192.168.0.100 and port 52460");
            Console.WriteLine("                Default: -ip:127.0.0.1:52460");
            Console.WriteLine("");
            Console.WriteLine("  -tty        Specifies the tty of the reader");
            Console.WriteLine("                Example: -tty:/dev/ttyACM0");
            Console.WriteLine("                -> Reader is at /dev/ttyACM0");
            Console.WriteLine("                Default: -tty:/dev/ttyACM0");
            Console.WriteLine("");
            Console.WriteLine("  -antenna    Specifies the antenna sequence");
            Console.WriteLine("                Example: -antenna:3,1,2");
            Console.WriteLine("                -> The antennas #3 #1 & #2 are scanned in this sequence");
            Console.WriteLine("                Default: -antenna:1,2,3,4");
            Console.WriteLine("");
            Console.WriteLine("  -time       Specifies how long each antenna is scanned");
            Console.WriteLine("                Example: -time:750");
            Console.WriteLine("                -> Each antenna is scanned for 750 msecs");
            Console.WriteLine("                Default: -time:500");
            Console.WriteLine("");
            Console.WriteLine("  -att        Specifies the attenuation for the scan");
            Console.WriteLine("                Example: -att:5");
            Console.WriteLine("                -> Every antenna is used with an attenuation of 5");
            Console.WriteLine("                Example: -att:0-5");
            Console.WriteLine("                -> Every antenna is used with an attenuation of 0, 1, 2, 3, 4, 5");
            Console.WriteLine("                Example: -att:0,3");
            Console.WriteLine("                -> Every antenna is used with an attenuation of 0, 3");
            Console.WriteLine("                Default: -att:0");
            Console.WriteLine("");
            Console.WriteLine("  -freq         Specifies the frequency for the scan");
            Console.WriteLine("                 0  = ETSI LOWER (865,7|866,3|866,9|867,5)");
            Console.WriteLine("                 1  = ETSI UPPER (916,3|917,5|918,7|919,9)");
            Console.WriteLine("                Example: -freq:0");
            Console.WriteLine("                -> The ETSI LOWER frequencies are used used");
            Console.WriteLine("                Example: -freq:0-1");
            Console.WriteLine("                -> All frequencies from ETSI LOWER to ETSI UPPER are used");
            Console.WriteLine("                Example: -freq:0,1");
            Console.WriteLine("                -> The frequencies ETSI LOWER and ETSI UPPER are used");
            Console.WriteLine("                Default: -freq:0,1");
            Console.WriteLine("");
            Console.WriteLine("  -lf         Specifies the link frequency for the scan");
            Console.WriteLine("                 0  =  40 kHz");
            Console.WriteLine("                 1  =  80 kHz");
            Console.WriteLine("                 2  = 160 kHz");
            Console.WriteLine("                 3  = 213 kHz");
            Console.WriteLine("                 4  = 256 kHz");
            Console.WriteLine("                 5  = 320 kHz");
            Console.WriteLine("                Example: -lf:0");
            Console.WriteLine("                -> The link frequency 40 kHz is used");
            Console.WriteLine("                Example: -lf:0-5");
            Console.WriteLine("                -> All link frequencies from 40kHz to 320kHz are used");
            Console.WriteLine("                Example: -lf:0,2");
            Console.WriteLine("                -> The link frequency 40kHz and 160kHz are used.");
            Console.WriteLine("                Default: -lf:2");
            Console.WriteLine("");
            Console.WriteLine("  -code       Specifies the coding for the scan");
            Console.WriteLine("                 0  =  FM0");
            Console.WriteLine("                 1  =  Miller2");
            Console.WriteLine("                 2  =  Miller4");
            Console.WriteLine("                 3  =  Miller8");
            Console.WriteLine("                Example: -code:0");
            Console.WriteLine("                -> The coding FM0 is used");
            Console.WriteLine("                Example: -code:0-3");
            Console.WriteLine("                -> All codings from FM0 to Miller8 are used");
            Console.WriteLine("                Example: -code:0,2");
            Console.WriteLine("                -> The coding FM0 and Miller4 are used.");
            Console.WriteLine("                Default: -code:1");
            Console.WriteLine("");
            Console.WriteLine("  -lfCode     Specifies a combination of link frequency and coding for the scan");
            Console.WriteLine("                Example: -lfCode:[0,1]");
            Console.WriteLine("                -> The link frequency 40kHz and the coding Miller2 is used");
            Console.WriteLine("                Example: -lfCode:[0,1]+[2,3]");
            Console.WriteLine("                -> The link frequency 40kHz and the coding Miller2 and 160kHz with Miller8 is used");
            Console.WriteLine("");
            Console.WriteLine("  -count      Specifies the estimated count of tags to read");
            Console.WriteLine("                Example: -count:10");
            Console.WriteLine("                -> The reader is optimal configured to read 10 tags");
            Console.WriteLine("                Default: -count:4");
            Console.WriteLine("");
            Console.WriteLine("  -rounds     Specifies the count of inventory rounds per select");
            Console.WriteLine("                Example: -rounds:3");
            Console.WriteLine("                -> The reader sends a select command every 3. inventory round");
            Console.WriteLine("                Default: -rounds:1");
            Console.WriteLine("");
            Console.WriteLine("  -rInfo      Specifies the amount of printed read infos");
            Console.WriteLine("                Example: -rInfo:1");
            Console.WriteLine("                -> Only max values are printed out");
            Console.WriteLine("                Default: -rInfo:1");
            Console.WriteLine("");
            Console.WriteLine("  -r          Will reset the reader immediately");
            Console.WriteLine("");
            Console.WriteLine(" Example:");
            Console.WriteLine("    .\\CSrfeTest_03_CmdLineScanner.exe -tty:COM7 -antenna:1 -att:3,6,9 -time:2000 -lfCode:[0,2]+[2,3] -rInfo:2 -rounds:2");
        }


        #region Reader Control

        /// <summary>
        /// Init reader and set all desired settings.
        /// </summary>
        /// <returns>Success</returns>
        private bool InitReader()
        {
            // Get reader ID
            uint readerId = 0;
            for (int i = 0; i < 2; i++)
            {
                if (_ProtocolHandler.getReaderID(out readerId))
                {
                    break;
                }
            }
            if (readerId == 0)
            {
                Console.WriteLine("ERROR: Communication.. GetReaderID");
                return false;
            }

            if (_ResetReader)
            {
                RebootReader();
                return false;
            }

            Constants.eRFE_CURRENT_SYSTEM currentSystem;
            if (!_ProtocolHandler.getCurrentSystem(out currentSystem))
            {
                Console.WriteLine("ERROR: Communication.. Get Current System " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            if (currentSystem != Constants.eRFE_CURRENT_SYSTEM.RFE_SYS_FIRMWARE)
            {
                Console.WriteLine("ERROR: Communication.. System Bootloader " + _ProtocolHandler.LastReturnCode.ToString());
                //m_protocolHandler.switchSystem();
                return false;
            }

            // Set epc size
            if (!_ProtocolHandler.setParam(0x23, new byte[] { 0x00 }))
            {
                Console.WriteLine("ERROR: Communication.. EPC Size " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            // Request antenna count
            byte antennaCount = 0;
            if (!_ProtocolHandler.getAntennaCount(out antennaCount))
            {
                Console.WriteLine("ERROR: Communication.. Antenna Count " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            // Check the antenna sequence
            List<ushort> newAntennaSequence = new List<ushort>();
            for (int i = 0; i < _AntennaSequence.Count; i++)
            {
                if (_AntennaSequence[i] <= antennaCount)
                {
                    newAntennaSequence.Add(_AntennaSequence[i]);
                }
                else
                {
                    Global.trc(1, "Skipping antennaId " + _AntennaSequence[i].ToString() + " because reader does not have this");
                }
            }
            _AntennaSequence = newAntennaSequence;

            // Activate heartbeat
            if (!_ProtocolHandler.setHeartBeat(Constants.eRFE_HEARTBEAT_SIGNAL.HEARTBEAT_ON, 150))
            {
                Console.WriteLine("ERROR: Communication.. HeartBeat " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            // Set Q
            byte init_q = 2;
            int val = 4;
            for (int i = 2; i < 15; i++)
            {
                if (val >= _EstimatedCount)
                {
                    init_q = (byte)i;
                    break;
                }
                val <<= 1;
            }
            if (!_ProtocolHandler.setParam(0x26, new byte[] { init_q, 2, 15 }))
            {
                Console.WriteLine("ERROR: Communication.. Q value" + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            if (!_ProtocolHandler.setParam(0x29, new byte[] { (byte)_InvRounds }))
            {
                Console.WriteLine("ERROR: Communication.. Inventory rounds" + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            return true;
        }

        /// <summary>
        /// Restarts the cyclic inventory procedure.
        /// </summary>
        /// <returns>TRUE if everything was OK; FALSE if any error happened</returns>
        private bool RestartCyclicInventory()
        {
            if (_FirstStart)
            {
                _CurrentAttenuationIndex = 0;
                _CurrentFrequencyIndex = 0;
                _CurrentAntennaSequenceIndex = 0;
                _CurrentLfCodingIndex = 0;
                _FirstStart = false;
            }
            else
            {
                _CurrentLfCodingIndex++;
                if (_CurrentLfCodingIndex >= _LfCoding.Count)
                {
                    _CurrentLfCodingIndex = 0;

                    _CurrentAttenuationIndex++;
                    if (_CurrentAttenuationIndex >= _Attenuation.Count)
                    {
                        _CurrentAttenuationIndex = 0;

                        _CurrentFrequencyIndex++;
                        if (_CurrentFrequencyIndex >= _Frequency.Count)
                        {
                            _CurrentFrequencyIndex = 0;

                            _CurrentAntennaSequenceIndex++;
                            if (_CurrentAntennaSequenceIndex >= _AntennaSequence.Count)
                            {
                                _CurrentAntennaSequenceIndex = 0;
                                PrintOutput();
                                return true;
                            }
                        }
                    }
                }
            }

            ushort antennaId = _AntennaSequence[_CurrentAntennaSequenceIndex];
            ushort attenuation = _Attenuation[_CurrentAttenuationIndex];
            ushort frequency = _Frequency[_CurrentFrequencyIndex];
            Tuple<ushort, ushort> lfCode = _LfCoding[_CurrentLfCodingIndex];

            Global.trc(0, " SCANNING " + string.Format("@{0,2} with {1,2}steps at {2} and {3},{4}", antennaId, attenuation, ToString_Frequency(frequency), ToString_LinkFrequency(lfCode.Item1), ToString_Coding(lfCode.Item2)));

            // Set coding
            if (!_ProtocolHandler.setParam(0x21, new byte[] { (byte)lfCode.Item2 }))
            {
                Console.WriteLine("ERROR: Communication.. Coding " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            // Set Link Frequency
            if (!_ProtocolHandler.setParam(0x20, new byte[] { (byte)lfCode.Item1 }))
            {
                Console.WriteLine("ERROR: Communication.. Link Frequency " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            // Set freqeuncy
            List<uint> freqList;
            switch (frequency)
            {
                default:
                case 0:
                    freqList = _EtsiLowerFrequencies;
                    break;
                case 1:
                    freqList = _EtsiUpperFrequencies;
                    break;
            }
            if (!_ProtocolHandler.setFrequency(0, freqList))
            {
                Console.WriteLine("ERROR: Communication.. Frequency " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }
            if (!_ProtocolHandler.setLbtParams(0, 0, (ushort)(_ScanTimeOut / freqList.Count), -40))
            {
                Console.WriteLine("ERROR: Communication.. Frequency (LBT) " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }


            // Set attenuation
            if (!_ProtocolHandler.setAttenuation(attenuation))
            {
                Console.WriteLine("ERROR: Communication.. Attenuation " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            // Set anntenna
            List<Tuple<byte, ulong>> ansequ = new List<Tuple<byte, ulong>>();
            ansequ.Add(new Tuple<byte, ulong>((byte)antennaId, 0xFFFFFFFF));
            if (!_ProtocolHandler.setAntennaSequence(ansequ))
            {
                Console.WriteLine("ERROR: Communication.. AntennaSequence: " + antennaId.ToString() + " " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            bool ok = false;
            for (int i = 0; i < 3; i++)
            {
                ok = _ProtocolHandler.setCyclicInventory(true, (ulong)_ScanTimeOut);
                if (ok)
                    break;
            }

            // Start cyclic inventory
            if (!ok)
            {
                Console.WriteLine("ERROR: Communication.. CyclicInventory " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            return true;
        }

        /// <summary>
        /// Reboots the reader in case of failure
        /// </summary>
        private void RebootReader()
        {
            _ProtocolHandler.reboot();
        }

        #endregion


        #region Event Handler

        /// <summary>
        /// Event handler for cyclic inventory event
        /// </summary>
        /// <param name="tagEvent">Happened tag event</param>
        public void onCyclicInventory(CSrfeProtocolHandler.TagEvent tagEvent)
        {
            Global.trc(3, "  -- cyclicInventory");
            int ant = 1;
            if (tagEvent.hasAntenna)
                ant = tagEvent.antennaId;

            Global.trc(2, "    " + tagEvent.ToString() + " @ " + ant.ToString());

            ushort antenna = _AntennaSequence[_CurrentAntennaSequenceIndex];
            ushort attenuation = _Attenuation[_CurrentAttenuationIndex];
            ushort frequency = _Frequency[_CurrentFrequencyIndex];
            Tuple<ushort, ushort> lfCode = _LfCoding[_CurrentLfCodingIndex];

            List<ReadData> data = null;
            bool contains = false;
            int pos = 0;

            if (_ResultData.ContainsKey(tagEvent.tagId))
                data = _ResultData[tagEvent.tagId];

            if (data == null)
            {
                data = new List<ReadData>();
            }
            else
            {
                for (int i = 0; i < data.Count; i++)
                {
                    if (data[i].AntennaID == antenna && data[i].Attenuation == attenuation && data[i].Frequency == frequency &&
                       data[i].LinkFrequency == lfCode.Item1 && data[i].Coding == lfCode.Item2)
                    {
                        contains = true;
                        pos = i;
                        break;
                    }
                }
            }

            if (contains)
            {
                ReadData p = data[pos];
                p.ReadCount++;
                data[pos] = p;
                Global.trc(3, "     update");
            }
            else
            {
                ReadData p = new ReadData();
                p.AntennaID = antenna;
                p.Attenuation = attenuation;
                p.Frequency = frequency;
                p.LinkFrequency = lfCode.Item1;
                p.Coding = lfCode.Item2;
                p.ReadCount = 1;
                data.Add(p);
                Global.trc(3, "     create");
            }

            _ResultData[tagEvent.tagId] = data;
        }

        /// <summary>
        /// Event handler for the heartbeat event
        /// </summary>
        /// <param name="data">Data sent with the heartbeat event</param>
        public void onHeartBeat(byte[] data)
        {
        }

        #endregion Event Handler


        #region String Helper

        static string ToString_Frequency(ushort freq)
        {
            switch (freq)
            {
                case 0: return "ETSI LOWER";
                case 1: return "ETSI UPPER";
                default: return "UNKOWN";
            }
        }

        static string ToString_LinkFrequency(ushort lf)
        {
            switch (lf)
            {
                case 0: return " 40kHz";
                case 1: return " 80kHz";
                case 2: return "160kHz";
                case 3: return "213kHz";
                case 4: return "256kHz";
                case 5: return "320kHz";
                default: return "UNKOWN";
            }
        }

        static string ToString_Coding(ushort code)
        {
            switch (code)
            {
                case 0: return "    FM0";
                case 1: return "Miller2";
                case 2: return "Miller4";
                case 3: return "Miller8";
                default: return "UNKOWN";
            }
        }

        #endregion String Helper


        #region Parse Helper

        /// <summary>
        /// Reads an integer value from the argument
        /// </summary>
        /// <param name="arg">Given argument</param>
        /// <param name="key">Key to identify the argument</param>
        /// <param name="min">Minimum allowed value</param>
        /// <param name="max">Maximum allwoed value</param>
        /// <returns>The parsed value.</returns>
        private static int GetValue(string arg, string key, int min, int max)
        {
            string str = arg.Remove(0, key.Length);

            int tmp = Int32.Parse(str);

            if (tmp < min || tmp > max)
                throw new FormatException("The given value " + tmp + " is out of range. [" + min + "," + max + "]");

            return tmp;
        }

        /// <summary>
        /// Reads a list of values from the argument 
        /// </summary>
        /// <param name="arg">Given argument</param>
        /// <param name="key">Key to identify the argument</param>
        /// <param name="min">Minimum allowed value</param>
        /// <param name="max">Maximum allwoed value</param>
        /// <returns>The parsed list of values.</returns>
        private static List<ushort> GetList(string arg, string key, ushort min, ushort max)
        {
            string str = arg.Remove(0, key.Length);

            if (str.Contains(","))
            {
                string[] list = str.Split(",".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                if (list.Length == 0)
                    throw new FormatException("The given list has 0 entries");

                List<ushort> tmpList = new List<ushort>();
                for (int i = 0; i < list.Length; i++)
                {
                    ushort tmp;
                    tmp = UInt16.Parse(list[i]);

                    if (tmp < min || tmp > max)
                        throw new FormatException("The given value " + tmp + " is out of range. [" + min + "," + max + "]");

                    tmpList.Add(tmp);
                }

                if (tmpList.Count == 0)
                    throw new FormatException("The calculated list has 0 entries");

                return tmpList;
            }
            else if (str.Contains("-"))
            {
                string[] list = str.Split("-".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                if (list.Length != 2)
                    throw new FormatException("A From-To value must have two values");

                int tmp_beg = 0;
                int tmp_end = 0;

                tmp_beg = UInt16.Parse(list[0]);
                tmp_end = UInt16.Parse(list[1]);

                if (tmp_beg >= tmp_end)
                    throw new FormatException("The given starting value " + tmp_beg + " is higher than the ending value " + tmp_end);

                if (tmp_beg < min || tmp_beg > max)
                    throw new FormatException("The given starting value " + tmp_beg + " is out of range. [" + min + "," + max + "]");

                if (tmp_end < min || tmp_end > max)
                    throw new FormatException("The given ending value " + tmp_end + " is out of range. [" + min + "," + max + "]");

                List<ushort> tmpList = new List<ushort>();

                for (int i = tmp_beg; i <= tmp_end; i++)
                    tmpList.Add((ushort)i);

                if (tmpList.Count == 0)
                    throw new FormatException("The calculated list has 0 entries");

                return tmpList;
            }
            else
            {
                ushort tmp = UInt16.Parse(str);

                if (tmp < min || tmp > max)
                    throw new FormatException("The given value " + tmp + " is out of range. [" + min + "," + max + "]");

                List<ushort> tmpList = new List<ushort>();
                tmpList.Add(tmp);

                return tmpList;
            }
        }

        /// <summary>
        /// Parses the given arguments
        /// </summary>
        /// <param name="args">Given arguments</param>
        /// <returns>TRUE if everything was OK; FALSE if any error happened</returns>
        private bool ParseArguments(string[] args)
        {
            List<ushort> linkFrequency = new List<ushort>();
            List<ushort> coding = new List<ushort>();

            //--------------------------------------------------------------------------------
            // Parse arguments
            foreach (string arg in args)
            {
                try
                {
                    if (arg.StartsWith("-trc:"))
                    {
                        string traceStr = arg.Remove(0, 5);
                        int tracLevel = UInt16.Parse(traceStr);

                        Global.m_tracer.TraceLevel = tracLevel;
                    }
                    else if (arg.StartsWith("-ip:"))
                    {
                        string ipStr = arg.Remove(0, 4);
                        string[] list = ipStr.Split(":".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                        if (list.Length != 2)
                            throw new FormatException("IP address format should be like: 192.168.0.100:52460");

                        string tmpIp = list[0];
                        string regEx = "[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}";
                        if (!Regex.IsMatch(tmpIp, regEx))
                            throw new FormatException("IP address format should be like: 192.168.0.100:52460");

                        ushort tmpPort = UInt16.Parse(list[1]);

                        _IP = tmpIp;
                        _Port = tmpPort;
                        _UseTCP = true;
                    }
                    else if (arg.StartsWith("-tty:"))
                    {
                        string ttyStr = arg.Remove(0, 5);

                        _TTY = ttyStr;
                        _UseTCP = false;
                    }
                    else if (arg.StartsWith("-antenna:"))
                    {
                        _AntennaSequence = GetList(arg, "-antenna:", 0, 32);
                    }
                    else if (arg.StartsWith("-att:"))
                    {
                        _Attenuation = GetList(arg, "-att:", 0, 19);
                    }
                    else if (arg.StartsWith("-freq:"))
                    {
                        _Frequency = GetList(arg, "-freq:", 0, 1);
                    }
                    else if (arg.StartsWith("-lf:"))
                    {
                        linkFrequency = GetList(arg, "-lf:", 0, 5);
                    }
                    else if (arg.StartsWith("-code:"))
                    {
                        coding = GetList(arg, "-code:", 0, 3);
                    }
                    else if (arg.StartsWith("-lfCode:"))
                    {
                        string str = arg.Remove(0, 8);
                        string[] list = str.Split("+".ToCharArray());
                        foreach (string part in list)
                        {
                            string s = part.Remove(0, 1);
                            s = s.Remove(s.Length - 1, 1);
                            string[] l = s.Split(",".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                            if (l.Length != 2)
                                throw new FormatException("A LF-Coding entry must consist of two parameters seperated by a ','. Example: [0,1]");

                            ushort tmp_lf = UInt16.Parse(l[0]);
                            ushort tmp_code = UInt16.Parse(l[1]);

                            if (tmp_lf < 0 || tmp_lf > 5)
                                throw new FormatException("The given link frequency value is out of range. [0-5]");
                            if (tmp_code < 0 || tmp_code > 3)
                                throw new FormatException("The given coding value is out of range. [0-3]");

                            _LfCoding.Add(new Tuple<ushort, ushort>(tmp_lf, tmp_code));
                        }
                    }
                    else if (arg.StartsWith("-time:"))
                    {
                        _ScanTimeOut = GetValue(arg, "-time:", 1, Int32.MaxValue);
                    }
                    else if (arg.StartsWith("-count:"))
                    {
                        _EstimatedCount = GetValue(arg, "-count:", 1, Int32.MaxValue);
                    }
                    else if (arg.StartsWith("-rounds:"))
                    {
                        _InvRounds = GetValue(arg, "-rounds:", 1, Byte.MaxValue);
                    }
                    else if (arg.StartsWith("-rInfo:"))
                    {
                        _PrintReadInfoLevel = GetValue(arg, "-rInfo:", 1, 4);
                    }
                    else if (arg.StartsWith("-out:"))
                    {
                        _OutputFile = arg.Remove(0, 5);
                    }
                    else if (arg.StartsWith("-r"))
                    {
                        _ResetReader = true;
                    }
                    else if (arg.StartsWith("-h"))
                    {
                        _PrintHelp = true;
                        break;
                    }
                }
                catch(Exception e)
                {
                    Console.WriteLine("ERROR: Error while parsing the argument: " + arg);
                    Console.WriteLine("   -> " + e.Message);
                    return false;
                }
            }

            if (_PrintHelp)
            {
                return true;
            }


            //--------------------------------------------------------------------------------
            // Check Input

            if (_AntennaSequence.Count == 0)
            {
                Console.WriteLine("ERROR: No valid antenna sequence");
                return false;
            }

            if (_Attenuation.Count == 0)
            {
                Console.WriteLine("ERROR: No valid attenuation");
                return false;
            }

            if (_Frequency.Count == 0)
            {
                Console.WriteLine("ERROR: No valid frequency");
                return false;
            }

            if (_OutputFile.Length > 0)
            {
                if (!File.Exists(_OutputFile))
                {
                    Console.WriteLine("ERROR: Can NOT open file: " + _OutputFile);
                    return false;
                }
            }


            //--------------------------------------------------------------------------------
            // Prepare values

            for (int i = 0; i < linkFrequency.Count; i++)
                for (int j = 0; j < coding.Count; j++)
                    _LfCoding.Add(new Tuple<ushort, ushort>(linkFrequency[i], coding[j]));

            if (_LfCoding.Count == 0)
            {
                _LfCoding.Add(new Tuple<ushort, ushort>(2, 1));
            }


            //--------------------------------------------------------------------------------
            // Trace settings

            Global.trc(0, "RF-Embedded RFID Command Line Scanner");
            Global.trc(0, "=====================================");
            Global.trc(0, " Reader:                  " + ((_UseTCP) ? (_IP + ":" + _Port.ToString()) : _TTY));

            string[] antList = new string[_AntennaSequence.Count]; ;
            for (int i = 0; i < _AntennaSequence.Count; i++)
                antList[i] = string.Format("{0}", _AntennaSequence[i]);

            string[] attList = new string[_Attenuation.Count]; ;
            for (int i = 0; i < _Attenuation.Count; i++)
                attList[i] = string.Format("{0}steps", _Attenuation[i]);

            string[] freqStr = new string[_Frequency.Count]; ;
            for (int i = 0; i < _Frequency.Count; i++)
                freqStr[i] = string.Format("{0}", ToString_Frequency(_Frequency[i]));

            string[] lfCodeList = new string[_LfCoding.Count]; ;
            for (int i = 0; i < _LfCoding.Count; i++)
            {
                Tuple<ushort, ushort> lfCode = _LfCoding[i];
                lfCodeList[i] = string.Format("[{0},{1}]", ToString_LinkFrequency(lfCode.Item1), ToString_Coding(lfCode.Item2));
            }

            Global.trc(0, " Antenna:                 " + string.Join(", ", antList));
            Global.trc(0, " Attenuation:             " + string.Join(", ", attList));
            Global.trc(0, " Frequency:               " + string.Join(", ", freqStr));
            Global.trc(0, " Link Frequency/Coding:   " + string.Join(", ", lfCodeList));
            Global.trc(0, " Count:                   " + _EstimatedCount.ToString());
            Global.trc(0, " Rounds:                  " + _InvRounds.ToString());
            Global.trc(0, " Time:                    " + _ScanTimeOut.ToString());
            Global.trc(0, " Output Info Level:       " + _PrintReadInfoLevel);
            Global.trc(0, " Output File:             " + _OutputFile);

            return true;
        }

        #endregion Parse Helper



        /// <summary>
        /// Comparer for byte[]
        /// </summary>
        private class ByteArrayComparer : IEqualityComparer<byte[]>
        {
            public bool Equals(byte[] left, byte[] right)
            {
                if (left == null || right == null)
                {
                    return left == right;
                }
                return left.SequenceEqual(right);
            }
            public int GetHashCode(byte[] key)
            {
                if (key == null)
                    throw new ArgumentNullException("key");
                return key.Sum(b => b);
            }
        }



        /// <summary>
        /// Main routine
        /// </summary>
        /// <param name="args">arguments</param>
        static int Main(string[] args)
        {
            CSrfeTest_CmdLineScanner p = new CSrfeTest_CmdLineScanner();
            if (!p.Init(args))
                return 0;
            return p.Execute();
        }
    }
}
