﻿/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using CSrfeReaderInterface;
using CSrfeReaderInterface.device;
using CSrfeReaderInterface.impl.device;
using CSrfeReaderInterface.protocol;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;

namespace CSrfeTest_CmdLineEpcWriter
{
    class CSrfeTest_CmdLineEpcWriter
    {
        private bool _UseTCP = true;
        private string _IP = "127.0.0.1";
        private ushort _Port = 52470;
        private string _TTY = "COM1";

        private bool _ResetReader = false;
        private bool _PrintHelp = false;

        private CSrfePURprotocolHandler _ProtocolHandler = null;

        private byte[] _Target = null;
        private byte[] _Password = new byte[4] { 0x00, 0x00, 0x00, 0x00 };

        private ushort _Antenna = 1;
        private ushort _Attenuation = 0;
        private ushort _Frequency = 0;
        private ushort _LinkFrequency = 2;
        private ushort _Coding = 1;

        private List<uint> _EtsiLowerFrequencies = new List<uint>();
        private List<uint> _EtsiUpperFrequencies = new List<uint>();

        private bool _Exit_Error = false;
        private bool _Exit_Success = false;

        private ushort _WordCountPerWrite = 1;

        private bool _TagFound = false;

        /// <summary>
        /// Constructor
        /// </summary>
        public CSrfeTest_CmdLineEpcWriter()
        {
            _ProtocolHandler = null;

            // Setup frequency lists
            // Setup lower ETSI frequencies
            _EtsiLowerFrequencies.Add(865700);
            _EtsiLowerFrequencies.Add(866300);
            _EtsiLowerFrequencies.Add(866900);
            _EtsiLowerFrequencies.Add(867500);
            // Setup upper ETSI frequencies
            _EtsiUpperFrequencies.Add(916300);
            _EtsiUpperFrequencies.Add(917500);
            _EtsiUpperFrequencies.Add(918700);
            _EtsiUpperFrequencies.Add(919900);
        }

        /// <summary>
        /// Initialize the reader with the given arguments
        /// </summary>
        /// <param name="args">Arguments of the command line</param>
        /// <returns>Returns if the initialisation was successful</returns>
        public bool Init(string[] args)
        {
            // create trace and turn off 
            Global.m_tracer = new CSrfeTest_ConsoleApplication.Trace.ConsoleTrace();
            Global.m_tracer.TraceLevel = 0;

            Global.trc(3, "  -- init");

            if (!ParseArguments(args))
                return false;

            if(_PrintHelp)
            {
                PrintHelp();
                return false;
            }

            // Create communication device
            IProtocolDeviceInterface device = null;
            if (_UseTCP)
            {
                IPAddress ipAddress = IPAddress.Parse("127.0.0.1");
                if (_IP != "")
                    IPAddress.Parse(_IP);

                device = new TCPDevice(ipAddress, _Port);
                if (!device.Open())
                {
                    Console.WriteLine("ERROR: Could not connect to IP " + _IP + ":" + _Port + ".");
                    return false;
                }
            }
            else
            {
                device = new SerialDevice(_TTY);
                if (!device.Open())
                {
                    Console.WriteLine("ERROR: Could not open " + _TTY + ".");
                    return false;
                }
            }
            

            // Create Protocol handler
            _ProtocolHandler = new CSrfePURprotocolHandler(device);
            _ProtocolHandler.HeartBeat += new CSrfeProtocolHandler.HeartBeatHandler(onHeartBeat);
            _ProtocolHandler.CyclicInventory += new CSrfeProtocolHandler.CyclicInventoryHandler(onCyclicInventory);

            return InitReader();
        }

        /// <summary>
        /// Main Function to handle the reader and start scanning
        /// </summary>
        /// <returns>The return code for Main</returns>
        public int Execute()
        {
            _ProtocolHandler.setAntennaPower(true);

            DateTimeOffset startTime = DateTimeOffset.Now;

            Global.trc(1, "Start");

            StartCyclicInventory();

            while (true)
            {
                if (DateTimeOffset.Now.Subtract(startTime).Seconds >= 30)
                {
                    Console.WriteLine("Timeout");
                    break;
                }

                if (_Exit_Error)
                {
                    Console.WriteLine("Exit on Error");
                    break;
                }

                if (_Exit_Success)
                {
                    Console.WriteLine("Success");
                    break;
                }

                Thread.Sleep(10);
            }

            _ProtocolHandler.setCyclicInventory(false);
            _ProtocolHandler.setAntennaPower(false);
            _ProtocolHandler.Close();

            Global.trc(3, "  -- exit");
            Global.trc(1, "Exiting application");

            return (_Exit_Success) ? 0 : -1;
        }
                
        /// <summary>
        /// Generates the help output.
        /// </summary>
        private void PrintHelp()
        {
            Console.WriteLine("");
            Console.WriteLine("");
            Console.WriteLine(" RF-Embedded RFID Command Line EPC Writer");
            Console.WriteLine(" ========================================");
            Console.WriteLine("");
            Console.WriteLine(" Parameters:");
            Console.WriteLine("  -target     Specifies the EPC of the tag after the write");
            Console.WriteLine("                Example: -target:020000000000000000000000");
            Console.WriteLine("                MANADTORY; no default");
            Console.WriteLine("");
            Console.WriteLine("  -pwd        Specifies the Password to use if the memory is locked");
            Console.WriteLine("                Example: -pwd:1234ABCD");
            Console.WriteLine("                -> The password 1234ABCD is used to write the epc");
            Console.WriteLine("                Default: -pwd:00000000");
            Console.WriteLine("");
            Console.WriteLine("  -ip         Specifies the ip and port of the reader");
            Console.WriteLine("                Example: -ip:192.168.0.100:52460");
            Console.WriteLine("                -> Reader is at IP 192.168.0.100 and port 52460");
            Console.WriteLine("                Default: -ip:127.0.0.1:52460");
            Console.WriteLine("");
            Console.WriteLine("  -tty        Specifies the tty of the reader");
            Console.WriteLine("                Example: -tty:/dev/ttyACM0");
            Console.WriteLine("                -> Reader is at /dev/ttyACM0");
            Console.WriteLine("                Default: -tty:/dev/ttyACM0");
            Console.WriteLine("");
            Console.WriteLine("  -antenna    Specifies the antenna");
            Console.WriteLine("                Example: -antenna:3");
            Console.WriteLine("                -> The antennas #3 is used");
            Console.WriteLine("                Default: -antenna:1");
            Console.WriteLine("");
            Console.WriteLine("  -att        Specifies the attenuation for the scan");
            Console.WriteLine("                Example: -att:5");
            Console.WriteLine("                -> Every antenna is used with an attenuation of 5");
            Console.WriteLine("                Default: -att:0");
            Console.WriteLine("");
            Console.WriteLine("  -freq         Specifies the frequency for the scan");
            Console.WriteLine("                 0  = ETSI LOWER (865,7|866,3|866,9|867,5)");
            Console.WriteLine("                 1  = ETSI UPPER (916,3|917,5|918,7|919,9)");
            Console.WriteLine("                Example: -freq:0");
            Console.WriteLine("                -> The ETSI LOWER frequencies are used used");
            Console.WriteLine("                Default: -freq:0");
            Console.WriteLine("");
            Console.WriteLine("  -lf         Specifies the link frequency for the scan");
            Console.WriteLine("                 0  =  40 kHz");
            Console.WriteLine("                 1  =  80 kHz");
            Console.WriteLine("                 2  = 160 kHz");
            Console.WriteLine("                 3  = 213 kHz");
            Console.WriteLine("                 4  = 256 kHz");
            Console.WriteLine("                 5  = 320 kHz");
            Console.WriteLine("                Example: -lf:0");
            Console.WriteLine("                -> The link frequency 40 kHz is used");
            Console.WriteLine("                Example: -lf:0-5");
            Console.WriteLine("                -> All link frequencies from 40kHz to 320kHz are used");
            Console.WriteLine("                Example: -lf:0,2");
            Console.WriteLine("                -> The link frequency 40kHz and 160kHz are used.");
            Console.WriteLine("                Default: -lf:2");
            Console.WriteLine("");
            Console.WriteLine("  -code       Specifies the coding for the scan");
            Console.WriteLine("                 0  =  FM0");
            Console.WriteLine("                 1  =  Miller2");
            Console.WriteLine("                 2  =  Miller4");
            Console.WriteLine("                 3  =  Miller8");
            Console.WriteLine("                Example: -code:0");
            Console.WriteLine("                -> The coding FM0 is used");
            Console.WriteLine("                Example: -code:0-3");
            Console.WriteLine("                -> All codings from FM0 to Miller8 are used");
            Console.WriteLine("                Example: -code:0,2");
            Console.WriteLine("                -> The coding FM0 and Miller4 are used.");
            Console.WriteLine("                Default: -code:1");
            Console.WriteLine("");
            Console.WriteLine("  -wc         Specifies the word count per write");
            Console.WriteLine("                Example: -wc:2");
            Console.WriteLine("                -> Per write the application tries to write 2 words");
            Console.WriteLine("                Default: -wc:1");
            Console.WriteLine("");
            Console.WriteLine("  -r          Will reset the reader immediately");
            Console.WriteLine("");
            Console.WriteLine(" Example:");
            Console.WriteLine("    .\\CSrfeTest_04_CmdLineEpcWriter.exe -tty:COM7 -wc:3 -target:1112131415161718191A1B1C");
        }


        #region Reader Control

        /// <summary>
        /// Init reader and set all desired settings.
        /// </summary>
        /// <returns>Success</returns>
        private bool InitReader()
        {
            // Get reader ID
            uint readerId = 0;
            for (int i = 0; i < 2; i++)
            {
                if (_ProtocolHandler.getReaderID(out readerId))
                {
                    break;
                }
            }
            if (readerId == 0)
            {
                Console.WriteLine("ERROR: Communication.. GetReaderID");
                return false;
            }

            if (_ResetReader)
            {
                RebootReader();
                return false;
            }

            Constants.eRFE_CURRENT_SYSTEM currentSystem;
            if (!_ProtocolHandler.getCurrentSystem(out currentSystem))
            {
                Console.WriteLine("ERROR: Communication.. Get Current System " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            if (currentSystem != Constants.eRFE_CURRENT_SYSTEM.RFE_SYS_FIRMWARE)
            {
                Console.WriteLine("ERROR: Communication.. System Bootloader " + _ProtocolHandler.LastReturnCode.ToString());
                //m_protocolHandler.switchSystem();
                return false;
            }

            // Request antenna count
            byte antennaCount = 0;
            if (!_ProtocolHandler.getAntennaCount(out antennaCount))
            {
                Console.WriteLine("ERROR: Communication.. Antenna Count " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }


            if (_Antenna > antennaCount)
            {
                Console.WriteLine("ERROR: Selected antenna is not available " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            // Restore Factory Settings
            if (!_ProtocolHandler.restoreFactorySettings())
            {
                Console.WriteLine("ERROR: Communication.. Restore Factory Settings " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            // Activate heartbeat
            if (!_ProtocolHandler.setHeartBeat(Constants.eRFE_HEARTBEAT_SIGNAL.HEARTBEAT_ON, 150))
            {
                Console.WriteLine("ERROR: Communication.. HeartBeat " + _ProtocolHandler.LastReturnCode.ToString());
                RebootReader();
                return false;
            }

            // Set freqeuncy
            List<uint> freqList;
            switch (_Frequency)
            {
                default:
                case 0:
                    freqList = _EtsiLowerFrequencies;
                    break;
                case 1:
                    freqList = _EtsiUpperFrequencies;
                    break;
            }
            if (!_ProtocolHandler.setFrequency(0, freqList))
            {
                Console.WriteLine("ERROR: Communication.. Frequency " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            if (!_ProtocolHandler.setLbtParams(0, 0, 1000, -40))
            {
                Console.WriteLine("ERROR: Communication.. Frequency (LBT) " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            // Set attenuation
            if (!_ProtocolHandler.setAttenuation(_Attenuation))
            {
                Console.WriteLine("ERROR: Communication.. Attenuation " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }

            // Set anntenna
            List<Tuple<byte, ulong>> ansequ = new List<Tuple<byte, ulong>>();
            ansequ.Add(new Tuple<byte, ulong>((byte)_Antenna, 0xFFFFFFFF));
            if (!_ProtocolHandler.setAntennaSequence(ansequ))
            {
                Console.WriteLine("ERROR: Communication.. AntennaSequence: " + _Antenna.ToString() + " " + _ProtocolHandler.LastReturnCode.ToString());
                return false;
            }


            List<Tuple<byte, byte[]>> Params = new List<Tuple<byte, byte[]>>();
            Params.Add(new Tuple<byte, byte[]>(0x00, new byte[] { 0x04 }));                    // Inventory Mode = Gen2 - Standard Single Tag
            Params.Add(new Tuple<byte, byte[]>(0x03, new byte[] { 0x02 }));                    // Tag-Id-Behavior = Send-Tag-Id-Immediately And Stop
            Params.Add(new Tuple<byte, byte[]>(0x20, new byte[] { (byte)_LinkFrequency }));    // Link Frequency
            Params.Add(new Tuple<byte, byte[]>(0x21, new byte[] { (byte)_Coding }));           // Coding
            Params.Add(new Tuple<byte, byte[]>(0x23, new byte[] { 0x00 }));                    // Gen2-EPC-Size = Dynamic
            Params.Add(new Tuple<byte, byte[]>(0x24, new byte[] { 0x01 }));                    // Gen2-Send-Handle = On
            Params.Add(new Tuple<byte, byte[]>(0x25, new byte[] { 0x01 }));                    // Gen2-Send-PC = On

            foreach (Tuple<byte, byte[]> param in Params)
            {
                if (!_ProtocolHandler.setParam(param.Item1, param.Item2))
                {
                    Console.WriteLine("ERROR: Setting Param 0x" + param.Item1.ToString("X2") + " to " + BitConverter.ToString(param.Item2) + " - " + _ProtocolHandler.LastReturnCode.ToString());
                    RebootReader();
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Restarts the cyclic inventory procedure.
        /// </summary>
        /// <returns>TRUE if everything was OK; FALSE if any error happened</returns>
        private bool StartCyclicInventory()
        {
            bool ok = false;
            for (int i = 0; i < 3; i++)
            {
                ok = _ProtocolHandler.setCyclicInventory(true);
                if (ok)
                    break;
            }

            // Start cyclic inventory
            if (!ok)
            {
                Console.WriteLine("ERROR: Communication.. CyclicInventory " + _ProtocolHandler.LastReturnCode.ToString());
                _Exit_Error = true;
                return false;
            }

            return true;
        }

        /// <summary>
        /// Reboots the reader in case of failure
        /// </summary>
        private void RebootReader()
        {
            _ProtocolHandler.reboot();
        }

        #endregion


        #region Event Handler

        /// <summary>
        /// Event handler for cyclic inventory event
        /// </summary>
        /// <param name="tagEvent">Happened tag event</param>
        public void onCyclicInventory(CSrfeProtocolHandler.TagEvent tagEvent)
        {
            Task.Factory.StartNew(() => { handleTag(tagEvent); });
        }

        /// <summary>
        /// Function to handle the tag event in another task, to not block the calling task.
        /// </summary>
        /// <param name="tagEvent">The tag event</param>
        public void handleTag(CSrfeProtocolHandler.TagEvent tagEvent)
        {
            Global.trc(3, "  -- cyclicInventory");
            Global.trc(1, "   " + tagEvent.ToString());

            if(!tagEvent.hasHandle && !tagEvent.hasPC)
            {
                Console.WriteLine("ERROR: Communication.. CyclicInventory Interrupt without Handle " + _ProtocolHandler.LastReturnCode.ToString());
                _Exit_Error = true;
                return;
            }

            if (!_TagFound)
            {
                //_Target = new byte[tagEvent.tagId.Length];
                //for(int i = 0; i < tagEvent.tagId.Length; i++)
                //{
                //    _Target[i] = (byte)(tagEvent.tagId[i] ^ (byte)0xAA);
                //}
                _TagFound = true;
            }

            // Set Length
            if (tagEvent.tagId.Length != _Target.Length)
            {
                byte wordCount = (byte)(_Target.Length / 2);

                byte t = tagEvent.pc[0];
                t &= 0x07;
                t |= (byte)((wordCount) << 3);
                tagEvent.pc[0] = t;

                bool ok = _ProtocolHandler.writeToHandle(tagEvent.handle, 0x01, 0x01, _Password, tagEvent.pc);

                Global.trc(1, "    -> Set Length  " + ((ok) ? "OK" : "ERR: " + _ProtocolHandler.LastReturnCode.ToString()));
            }
            // Set EPC
            else if(!(new ByteArrayComparer()).Equals(tagEvent.tagId, _Target))
            {
                ushort firstDiffWordPosition = 0;
                for (int i = 0; i < tagEvent.tagId.Length; i++)
                {
                    if (tagEvent.tagId[i] != _Target[i])
                    {
                        firstDiffWordPosition = (ushort)(i / 2);
                        break;
                    }
                }

                ushort wordsLeft = (ushort)((_Target.Length >> 1) - firstDiffWordPosition);
                ushort words2Write = _WordCountPerWrite;
                if (words2Write > wordsLeft)
                    words2Write = wordsLeft;

                byte[] data = new byte[words2Write * 2];
                Array.Copy(_Target, firstDiffWordPosition * 2, data, 0, words2Write * 2);

                bool ok = _ProtocolHandler.writeToHandle(tagEvent.handle, 0x01, (ushort)(0x02 + firstDiffWordPosition), _Password, data);

                Global.trc(1, "    -> Set EPC @" + firstDiffWordPosition.ToString("X2") + " to " + BitConverter.ToString(data) + "  " + ((ok) ? "OK" : "ERR: " + _ProtocolHandler.LastReturnCode.ToString()));
            }
            // Finished
            else
            {
                Global.trc(1, "    -> Match -> Finished!");
                _Exit_Success = true;
                return;
            }

            StartCyclicInventory();
        }

        /// <summary>
        /// Event handler for the heartbeat event
        /// </summary>
        /// <param name="data">Data sent with the heartbeat event</param>
        public void onHeartBeat(byte[] data)
        {
        }

        #endregion Event Handler


        #region String Helper

        static string ToString_Frequency(ushort freq)
        {
            switch (freq)
            {
                case 0: return "ETSI LOWER";
                case 1: return "ETSI UPPER";
                default: return "UNKOWN";
            }
        }

        static string ToString_LinkFrequency(ushort lf)
        {
            switch (lf)
            {
                case 0: return " 40kHz";
                case 1: return " 80kHz";
                case 2: return "160kHz";
                case 3: return "213kHz";
                case 4: return "256kHz";
                case 5: return "320kHz";
                default: return "UNKOWN";
            }
        }

        static string ToString_Coding(ushort code)
        {
            switch (code)
            {
                case 0: return "    FM0";
                case 1: return "Miller2";
                case 2: return "Miller4";
                case 3: return "Miller8";
                default: return "UNKOWN";
            }
        }

        #endregion String Helper


        #region Parse Helper

        /// <summary>
        /// Reads an integer value from the argument
        /// </summary>
        /// <param name="arg">Given argument</param>
        /// <param name="key">Key to identify the argument</param>
        /// <param name="min">Minimum allowed value</param>
        /// <param name="max">Maximum allwoed value</param>
        /// <returns>The parsed value.</returns>
        private static int GetValue(string arg, string key, int min, int max)
        {
            string str = arg.Remove(0, key.Length);

            int tmp = Int32.Parse(str);

            if (tmp < min || tmp > max)
                throw new FormatException("The given value " + tmp + " is out of range. [" + min + "," + max + "]");

            return tmp;
        }

        /// <summary>
        /// Reads a list of values from the argument 
        /// </summary>
        /// <param name="arg">Given argument</param>
        /// <param name="key">Key to identify the argument</param>
        /// <param name="min">Minimum allowed value</param>
        /// <param name="max">Maximum allwoed value</param>
        /// <returns>The parsed list of values.</returns>
        private static List<ushort> GetList(string arg, string key, ushort min, ushort max)
        {
            string str = arg.Remove(0, key.Length);

            if (str.Contains(","))
            {
                string[] list = str.Split(",".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                if (list.Length == 0)
                    throw new FormatException("The given list has 0 entries");

                List<ushort> tmpList = new List<ushort>();
                for (int i = 0; i < list.Length; i++)
                {
                    ushort tmp;
                    tmp = UInt16.Parse(list[i]);

                    if (tmp < min || tmp > max)
                        throw new FormatException("The given value " + tmp + " is out of range. [" + min + "," + max + "]");

                    tmpList.Add(tmp);
                }

                if (tmpList.Count == 0)
                    throw new FormatException("The calculated list has 0 entries");

                return tmpList;
            }
            else if (str.Contains("-"))
            {
                string[] list = str.Split("-".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                if (list.Length != 2)
                    throw new FormatException("A From-To value must have two values");

                int tmp_beg = 0;
                int tmp_end = 0;

                tmp_beg = UInt16.Parse(list[0]);
                tmp_end = UInt16.Parse(list[1]);

                if (tmp_beg >= tmp_end)
                    throw new FormatException("The given starting value " + tmp_beg + " is higher than the ending value " + tmp_end);

                if (tmp_beg < min || tmp_beg > max)
                    throw new FormatException("The given starting value " + tmp_beg + " is out of range. [" + min + "," + max + "]");

                if (tmp_end < min || tmp_end > max)
                    throw new FormatException("The given ending value " + tmp_end + " is out of range. [" + min + "," + max + "]");

                List<ushort> tmpList = new List<ushort>();

                for (int i = tmp_beg; i <= tmp_end; i++)
                    tmpList.Add((ushort)i);

                if (tmpList.Count == 0)
                    throw new FormatException("The calculated list has 0 entries");

                return tmpList;
            }
            else
            {
                ushort tmp = UInt16.Parse(str);

                if (tmp < min || tmp > max)
                    throw new FormatException("The given value " + tmp + " is out of range. [" + min + "," + max + "]");

                List<ushort> tmpList = new List<ushort>();
                tmpList.Add(tmp);

                return tmpList;
            }
        }

        /// <summary>
        /// Parses a byte array from a hex string
        /// </summary>
        /// <param name="hexString">Hex string</param>
        /// <returns>The parsed byte array</returns>
        private static byte[] ParseHexString(string hexString)
        {
            if (hexString.Length % 2 != 0)
                throw new FormatException("The given hex string " + hexString + " has wrong length.");

            byte[] result = new byte[hexString.Length / 2];
            for(int i = 0; i < hexString.Length/2; i++)
            {
                result[i] = (byte)Convert.ToInt32(hexString.Substring(i * 2, 2), 16);
            }

            return result;
        }

        /// <summary>
        /// Parses the given arguments
        /// </summary>
        /// <param name="args">Given arguments</param>
        /// <returns>TRUE if everything was OK; FALSE if any error happened</returns>
        private bool ParseArguments(string[] args)
        {
            //--------------------------------------------------------------------------------
            // Parse arguments
            foreach (string arg in args)
            {
                try
                {
                    if (arg.StartsWith("-trc:"))
                    {
                        string traceStr = arg.Remove(0, 5);
                        int tracLevel = UInt16.Parse(traceStr);

                        Global.m_tracer.TraceLevel = tracLevel;
                    }
                    else if (arg.StartsWith("-target:"))
                    {
                        string targetStr = arg.Remove(0, 8);
                        _Target = ParseHexString(targetStr);
                    }
                    else if (arg.StartsWith("-pwd:"))
                    {
                        string pwdStr = arg.Remove(0, 5);
                        _Password = ParseHexString(pwdStr);
                        if (_Password.Length != 4)
                            throw new FormatException("The provided password has wrong length (" + _Password.Length.ToString() + " != 4)");
                    }
                    else if (arg.StartsWith("-ip:"))
                    {
                        string ipStr = arg.Remove(0, 4);
                        string[] list = ipStr.Split(":".ToCharArray(), StringSplitOptions.RemoveEmptyEntries);
                        if (list.Length != 2)
                            throw new FormatException("IP address format should be like: 192.168.0.100:52460");

                        string tmpIp = list[0];
                        string regEx = "[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}";
                        if (!Regex.IsMatch(tmpIp, regEx))
                            throw new FormatException("IP address format should be like: 192.168.0.100:52460");

                        ushort tmpPort = UInt16.Parse(list[1]);

                        _IP = tmpIp;
                        _Port = tmpPort;
                        _UseTCP = true;
                    }
                    else if (arg.StartsWith("-tty:"))
                    {
                        string ttyStr = arg.Remove(0, 5);

                        _TTY = ttyStr;
                        _UseTCP = false;
                    }
                    else if (arg.StartsWith("-antenna:"))
                    {
                        _Antenna = (ushort)GetValue(arg, "-antenna:", 0, 32);
                    }
                    else if (arg.StartsWith("-att:"))
                    {
                        _Attenuation = (ushort)GetValue(arg, "-att:", 0, 19);
                    }
                    else if (arg.StartsWith("-freq:"))
                    {
                        _Frequency = (ushort)GetValue(arg, "-freq:", 0, 1);
                    }
                    else if (arg.StartsWith("-lf:"))
                    {
                        _LinkFrequency = (ushort)GetValue(arg, "-lf:", 0, 5);
                    }
                    else if (arg.StartsWith("-code:"))
                    {
                        _Coding = (ushort)GetValue(arg, "-code:", 0, 3);
                    }
                    else if (arg.StartsWith("-wc:"))
                    {
                        _WordCountPerWrite = (ushort)GetValue(arg, "-wc:", 1, 32);
                    }
                    else if (arg.StartsWith("-r"))
                    {
                        _ResetReader = true;
                    }
                    else if (arg.StartsWith("-h"))
                    {
                        _PrintHelp = true;
                        break;
                    }
                }
                catch(Exception e)
                {
                    Console.WriteLine("ERROR: Error while parsing the argument: " + arg);
                    Console.WriteLine("   -> " + e.Message);
                    return false;
                }
            }

            if (_PrintHelp)
            {
                return true;
            }


            //--------------------------------------------------------------------------------
            // Check Input

            if (_Target == null)
            {
                Console.WriteLine("ERROR: No valid target EPC");
                return false;
            }
            if ( (_Target.Length%2) != 0)
            {
                Console.WriteLine("ERROR: Target EPC of wrong size");
                return false;
            }

            //--------------------------------------------------------------------------------
            // Trace settings

            Global.trc(0, " RF-Embedded RFID Command Line EPC Writer");
            Global.trc(0, " ========================================");
            Global.trc(0, " Reader:                  " + ((_UseTCP) ? (_IP + ":" + _Port.ToString()) : _TTY));
            Global.trc(0, " Target:                  " + BitConverter.ToString(_Target));
            Global.trc(0, " Password:                " + BitConverter.ToString(_Password));
            Global.trc(0, " Antenna:                 " + _Antenna.ToString());
            Global.trc(0, " Attenuation:             " + _Attenuation.ToString());
            Global.trc(0, " Frequency:               " + ToString_Frequency(_Frequency));
            Global.trc(0, " Link Frequency:          " + ToString_LinkFrequency(_LinkFrequency));
            Global.trc(0, " Coding:                  " + ToString_Coding(_Coding));

            return true;
        }

        #endregion Parse Helper



        /// <summary>
        /// Comparer for byte[]
        /// </summary>
        private class ByteArrayComparer : IEqualityComparer<byte[]>
        {
            public bool Equals(byte[] left, byte[] right)
            {
                if (left == null || right == null)
                {
                    return left == right;
                }
                return left.SequenceEqual(right);
            }
            public int GetHashCode(byte[] key)
            {
                if (key == null)
                    throw new ArgumentNullException("key");
                return key.Sum(b => b);
            }
        }



        /// <summary>
        /// Main routine
        /// </summary>
        /// <param name="args">arguments</param>
        static int Main(string[] args)
        {
            CSrfeTest_CmdLineEpcWriter p = new CSrfeTest_CmdLineEpcWriter();
            if (!p.Init(args))
                return 0;
            return p.Execute();
        }
    }
}
