/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "QrfePURprotocolHandler.h"
#include "../QrfeReaderInterface.h"

using namespace QrfeReaderInterface;

/**
 * @brief QrfePURprotocolHandler::QrfePURprotocolHandler constructs a PUR implementation of the protocol handler
 * @param device    The communication device
 * @param parent    Parent object
 */
QrfePURprotocolHandler::QrfePURprotocolHandler(QIODevice *device, QObject* parent)
    : QrfeProtocolHandler(device, parent)
{
}

/**
 * @brief QrfePURprotocolHandler::postDetectReadParams retrieves the settings for PostDetect-READ from the reader
 * @param on        Indicates if the PostDetect-READ is turned ON
 * @param memBank   The memory bank to read from
 * @param address   The address to read from
 * @param size      The size of data to read
 * @param password  The password used for read
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::postDetectReadParams ( bool &on, uchar &memBank, ushort &address, uchar &size, QByteArray& password )
{
    QByteArray postDetectREAD;
    bool ret = getParam(0x06, postDetectREAD);
    if(!ret)
        return ret;

    if(postDetectREAD.size() != 9)
        return false;

    on 			= (postDetectREAD.at(0) != 0);
    memBank 	= postDetectREAD.at(1);
    address 	= 0;
    address		+= (ushort)( ((uchar)postDetectREAD.at(2)) << 8);
    address		+= (ushort)( ((uchar)postDetectREAD.at(3)) << 0);
    size	 	= postDetectREAD.at(4);
    password 	= postDetectREAD.mid(5, 4);
    return true;
}

/**
 * @brief QrfePURprotocolHandler::setPostDetectReadParams sets the params for the PostDetect-READ
 * @param on        Indicates if the PostDetect-READ is turned ON
 * @param memBank   The memory bank to read from
 * @param address   The address to read from
 * @param size      The size of data to read
 * @param password  The password used for read
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::setPostDetectReadParams ( bool on, uchar memBank, ushort address, uchar size, QByteArray password )
{
    QByteArray postDetectREAD;
    postDetectREAD.append(on);
    postDetectREAD.append(memBank);
    postDetectREAD.append((uchar)(address >> 8));
    postDetectREAD.append((uchar)(address >> 0));
    postDetectREAD.append(size);
    postDetectREAD.append(password);

    bool ret = setParam(0x06, postDetectREAD);
    if(!ret)
        return false;

    return true;
}

/**
 * @brief QrfePURprotocolHandler::blockWriteToTag writes data to a tag using the Gen2-BlockWrite command
 * @param tagId     TagId of the tag
 * @param mem_bank  Memory bank where to read data from
 * @param address   Address within the memory bank
 * @param passwd    The access password to read from the tag
 * @param blockSize The block size of BlockWrite that should be used in words
 * @param data      The data should be written
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::blockWriteToTag ( QByteArray tagId, uchar mem_bank, ushort address, QByteArray passwd, uchar blockSize, QByteArray data )
{
    if (passwd.size() != 4)
    {
        Global::trc(1, "BlockWrite to tag - NOK - Data");
        return false;
    }

    if (data.size() > 220)
    {
        Global::trc(1, "BlockWrite to tag - NOK - Data");
        return false;
    }

    QByteArray payload;
    payload.append((uchar) tagId.size());
    payload.append(tagId);
    payload.append((uchar) mem_bank);
    payload.append((uchar) (address >> 8));
    payload.append((uchar) address);
    payload.append(passwd);
    payload.append((uchar) blockSize);
    payload.append((uchar) data.size());
    payload.append(data);

    QByteArray resultData;

    if (!customTagCommand((uchar)0x03, payload, resultData))
        return false;

    return true;
}

/**
 * @brief QrfePURprotocolHandler::getTagHandle retrieves the handle from the given tag
 * @param tagId     TagId of the tag
 * @param handle    The retrieved handle
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::getTagHandle ( QByteArray tagId, QByteArray& handle )
{
    QByteArray payload;
    payload.append((uchar) tagId.size());
    payload.append(tagId);

    QByteArray resultData;

    if (!customTagCommand((uchar)0x04, payload, resultData))
        return false;

    if(resultData.size() < 2)
        return false;

    handle = resultData.mid(0, 2);

    return true;
}

/**
 * @brief QrfePURprotocolHandler::readFromHandle reads data from a tag using the handle
 * @param handle    Handle of the tag
 * @param mem_bank  Memory bank where to read data from
 * @param address   Address within the memory bank
 * @param passwd    The access password to read from the tag
 * @param count     The count of data that should be read
 * @param data      The read data
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::readFromHandle ( QByteArray handle, uchar mem_bank, ushort address, QByteArray passwd, uchar count, QByteArray& data )
{
    if (handle.size() != 2)
    {
        Global::trc(1, "Read from handle - NOK - Data");
        return false;
    }
    if (passwd.size() != 4)
    {
        Global::trc(1, "Read from handle - NOK - Data");
        return false;
    }

    QByteArray payload;
    payload.append(handle);
    payload.append((uchar) mem_bank);
    payload.append((uchar) (address >> 8));
    payload.append((uchar) address);
    payload.append(passwd);
    payload.append(count);

    QByteArray resultData;

    if (!customTagCommand((uchar)0x05, payload, resultData))
        return false;

    data = resultData.mid(1);

    return true;
}

/**
 * @brief QrfePURprotocolHandler::writeToHandle writes data to a tag using the handle
 * @param handle    Handle of the tag
 * @param mem_bank  Memory bank where to read data from
 * @param address   Address within the memory bank
 * @param passwd    The access password to read from the tag
 * @param data      The data that should be written
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::writeToHandle ( QByteArray handle, uchar mem_bank, ushort address, QByteArray passwd, QByteArray data )
{
    if (handle.size() != 2)
    {
        Global::trc(1, "Write to handle - NOK - Data");
        return false;
    }
    if (passwd.size() != 4)
    {
        Global::trc(1, "Write to handle - NOK - Data");
        return false;
    }
    if (data.size() > 220)
    {
        Global::trc(1, "Write to tag - NOK - Data");
        return false;
    }

    QByteArray payload;
    payload.append(handle);
    payload.append((uchar) mem_bank);
    payload.append((uchar) (address >> 8));
    payload.append((uchar) address);
    payload.append(passwd);
    payload.append((uchar) data.size());
    payload.append(data);

    QByteArray resultData;

    if (!customTagCommand((uchar)0x06, payload, resultData))
        return false;

    return true;
}

/**
 * @brief QrfePURprotocolHandler::blockWriteToHandle writes data to a tag using the handle and the BlockWrite command
 * @param handle    Handle of the tag
 * @param mem_bank  Memory bank where to read data from
 * @param address   Address within the memory bank
 * @param passwd    The access password to read from the tag
 * @param blockSize The block size of BlockWrite that should be used in words
 * @param data      The data that should be written
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::blockWriteToHandle ( QByteArray handle, uchar mem_bank, ushort address, QByteArray passwd, uchar blockSize, QByteArray data )
{
    if (handle.size() != 2)
    {
        Global::trc(1, "BlockWrite to handle - NOK - Data");
        return false;
    }
    if (passwd.size() != 4)
    {
        Global::trc(1, "BlockWrite to handle - NOK - Data");
        return false;
    }
    if (data.size() > 220)
    {
        Global::trc(1, "BlockWrite to tag - NOK - Data");
        return false;
    }

    QByteArray payload;
    payload.append(handle);
    payload.append((uchar) mem_bank);
    payload.append((uchar) (address >> 8));
    payload.append((uchar) address);
    payload.append(passwd);
    payload.append((uchar) blockSize);
    payload.append((uchar) data.size());
    payload.append(data);

    QByteArray resultData;

    if (!customTagCommand((uchar)0x08, payload, resultData))
        return false;

    return true;
}


/**
 * @brief QrfePURprotocolHandler::customGen2Command sends a custom tag command to the tag
 * @param handle                Handle of the tag
 * @param command               The two byte command to be sent
 * @param passwd                The passwd to be used
 * @param txBitCount            The count of bits to be sent
 * @param txBits                The bits to be sent
 * @param headerBit             Flag to signalize, if header bit was received
 * @param rxBytes               The bytes received from the tag
 * @return Success of the operation
 */
bool QrfePURprotocolHandler::customGen2Command ( QByteArray handle, QByteArray command, QByteArray passwd, uchar txBitCount, QByteArray txBits, bool& headerBit, QByteArray& rxBytes )
{
    if (handle.size() != 2)
    {
        Global::trc(1, "Custom Tag Command Handle - NOK - Data");
        return false;
    }
    if (command.size() != 2 || command.size() != 4)
    {
        Global::trc(1, "Custom Tag Command Handle - NOK - Data");
        return false;
    }
    if (passwd.size() != 4)
    {
        Global::trc(1, "Custom Tag Command Handle - NOK - Data");
        return false;
    }

    uchar txByteCount = (uchar)(txBitCount >> 3);
    if ((txBitCount % 8) > 0)
        txByteCount++;

    if (txByteCount != txBits.size())
    {
        Global::trc(1, "Custom Tag Command Handle - NOK - Data");
        return false;
    }

    QByteArray payload;
    payload.append(handle);
    payload.append(command);
    payload.append(passwd);
    payload.append((uchar) txBitCount);
    payload.append(txBits);

    QByteArray resultData;

    if (!customTagCommand((uchar)0x07, payload, resultData))
        return false;

    headerBit = ((uchar)resultData.at(0)) != 0;
    rxBytes = resultData.mid(2, resultData.size()-2);

    return true;
}
