/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "QrfeTagEvent.h"

#include <QStringList>

using namespace QrfeReaderInterface;

/**
 * @brief QrfeTagEvent::QrfeTagEvent constructs a new default tag event
 */
QrfeTagEvent::QrfeTagEvent()
{
    hasAntenna = false;
    hasRSSI = false;
    hasReadFrequency = false;
    hasMemory = false;
    hasTrigger = false;
    hasHandle = false;
    hasState = false;
    hasBattery = false;
    hasPC = false;
    hasMessageID = false;
    hasApplicationInfo = false;
}

/**
 * @brief QrfeTagEvent::QrfeTagEvent is the copy consturctor to copy the tag event
 * @param other The object to copy from
 */
QrfeTagEvent::QrfeTagEvent(const QrfeTagEvent& other)
{
    tagId = other.tagId;

    hasAntenna = other.hasAntenna;
    if (hasAntenna)
    {
        antennaId = other.antennaId;
    }

    hasRSSI = other.hasRSSI;
    if (hasRSSI)
    {
        rssi = other.rssi;
    }

    hasReadFrequency = other.hasReadFrequency;
    if (hasReadFrequency)
    {
        readFrequency = other.readFrequency;
    }

    hasMemory = other.hasMemory;
    if (hasMemory)
    {
        memBank = other.memBank;
        memAddr = other.memAddr;
        memData = other.memData;
    }

    hasTrigger = other.hasTrigger;
    if (hasTrigger)
    {
        trigger = other.trigger;
    }

    hasHandle = other.hasHandle;
    if (hasHandle)
    {
        handle = other.handle;
    }

    hasState = other.hasState;
    if (hasState)
    {
        state = other.state;
    }

    hasBattery = other.hasBattery;
    if (hasBattery)
    {
        battery = other.battery;
    }

    hasPC = other.hasPC;
    if (hasPC)
    {
        pc = other.pc;
    }

    hasMessageID = other.hasMessageID;
    if (hasMessageID)
    {
        messageID = other.messageID;
    }

    hasApplicationInfo = other.hasApplicationInfo;
    if (hasApplicationInfo)
    {
        applicationInfo = other.applicationInfo;
    }
}

/**
 * @brief QrfeTagEvent::toString returns the tag event presented in a string
 * @return the tag event human readable
 */
QString QrfeTagEvent::toString() const
{
    QStringList ret;

    ret << tagId.toHex();

    if (hasAntenna)
    {
        ret << QString("Antenna #%1").arg(antennaId);
    }

    if (hasRSSI)
    {
        ret << QString("RSSI %1").arg(QString(rssi.toHex()));
    }

    if (hasReadFrequency)
    {
        ret << QString("Frequency %1").arg(readFrequency);
    }

    if (hasMemory)
    {
        ret << QString("Data %1@%2 %3").arg(memBank).arg(memAddr).arg(QString(memData.toHex()));
    }

    if (hasTrigger)
    {
        ret << QString("Trigger %1").arg(trigger);
    }

    if (hasHandle)
    {
        ret << QString("Handle %1").arg(QString(handle.toHex()));
    }

    if (hasState)
    {
        ret << QString("State %1").arg(state);
    }

    if (hasBattery)
    {
        ret << QString("Battery %1").arg(battery);
    }

    if (hasPC)
    {
        ret << QString("PC %1").arg(QString(pc.toHex()));
    }

    if (hasMessageID)
    {
        ret << QString("MessageID %1").arg(messageID);
    }

    if (hasApplicationInfo)
    {
        ret << QString("AppInfo %1").arg(QString(applicationInfo.toHex()));
    }

    return ret.join(", ");
}
