/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "TestWidget.h"
#include "ui_TestWidget.h"

#include <QTcpSocket>
#include <QHostAddress>
#include <QSerialPort>
#include <QInputDialog>
#include <QTimer>

#include "impl/ConsoleTrace.h"
#include "impl/Sleeper.h"

#include <QrfeReaderInterface.h>
#include <protocol/QrfeProtocolHandler.h>
#include <protocol/QrfePURprotocolHandler.h>
#include <protocol/QrfeTagEvent.h>
#include <protocol/QrfeProtocolConstants.h>

TestWidget::TestWidget(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::TestWidget)
{
    ui->setupUi(this);

    m_ph = 0;
    m_trc = 0;

    m_currentTestState = IDLE;
    m_lastTagEventValid = true;

    ui->testGroupBox->setEnabled(false);

    connect(ui->startPushButton,    SIGNAL(clicked()),  this,   SLOT(startStopExit()));
}

TestWidget::~TestWidget()
{
    delete ui;
    if(m_ph != 0)delete m_ph;
    if(m_trc != 0)delete m_trc;
}


void TestWidget::init(QString serialPort)
{
    // Open serial port
    QSerialPort* port = new QSerialPort();
    port->setPortName(serialPort);
    port->setFlowControl(QSerialPort::NoFlowControl);
    port->setParity(QSerialPort::NoParity);
    port->setDataBits(QSerialPort::Data8);
    port->setStopBits(QSerialPort::OneStop);
    port->setBaudRate(QSerialPort::Baud115200);
    if(!port->open(QIODevice::ReadWrite))
    {
        print("Could not open the serial port " + serialPort);
        delete port;
        return;
    }

    init(port);
}

void TestWidget::init(QString ip, ushort port)
{
    // Open socket
    QTcpSocket* sock = new QTcpSocket();
    sock->connectToHost(QHostAddress(ip), port);
    if(!sock->waitForConnected())
    {
        print("Could not connect to " + ip + ":" + QString::number(port));
        delete sock;
        return;
    }

    init(sock);
}

void TestWidget::init(QIODevice *dev)
{
    bool ok = false;

    // Get implementation
    QString impl = QInputDialog::getItem(this, "Implementation", "Select implementation:", QStringList() << "Standard" << "PUR" << "AUR", 0, false, &ok);
    if(!ok || impl.isNull())
        return;

    // Create Tracer
    m_trc = new ConsoleTrace();
    QrfeReaderInterface::Global::m_tracer = m_trc;
    QrfeReaderInterface::Global::m_tracer->setTraceLevel(1);

    // Create Protocol Handler
    if(impl == "Standard")
    {
        m_ph = new QrfeReaderInterface::QrfeProtocolHandler(dev);
    }
    else if(impl == "PUR")
    {
        m_ph = new QrfeReaderInterface::QrfePURprotocolHandler(dev);
    }
    else if(impl == "AUR")
    {
        m_ph = new QrfeReaderInterface::QrfeProtocolHandler(dev);
    }

    m_ph->setHeartBeat(QrfeReaderInterface::HEARTBEAT_OFF);

    // Connect signals
    connect(m_ph, SIGNAL(heartBeat()),
            this,   SLOT(heartBeat()));
    connect(m_ph, SIGNAL(cyclicInventory(QrfeReaderInterface::QrfeTagEvent)),
            this,   SLOT(cyclicInventory(QrfeReaderInterface::QrfeTagEvent)));
    connect(m_ph, SIGNAL(stateChanged(int)),
            this,   SLOT(stateChanged(int)));
    connect(m_ph, SIGNAL(statusRegisterChanged(qulonglong)),
            this,   SLOT(statusRegisterChanged(qulonglong)));
    connect(m_ph, SIGNAL(gpioValuesChanged(ulong)),
            this,   SLOT(gpioValuesChanged(ulong)));

    // Prepare GUI
    clear();
    printMenu();
    ui->testGroupBox->setEnabled(true);
    ui->spinBox->setFocus(Qt::TabFocusReason);
    ui->startPushButton->setDefault(true);
}


void TestWidget::startStopExit()
{
    if(m_ph == 0)
        return;


    int states = 1;

    switch(m_currentTestState)
    {
    case IDLE:
        clear();

        ui->spinBox->setEnabled(false);
        ui->startPushButton->setEnabled(false);
        qApp->processEvents();

        m_currentTestRoutine = ui->spinBox->value();

        test(m_currentTestRoutine, true, states);

        ui->startPushButton->setEnabled(true);
        qApp->processEvents();

        if(states == 1)
        {
            m_currentTestState = FINISHED;
            ui->startPushButton->setText("Exit");
        }
        else
        {
            m_currentTestState = RUNNING;
            ui->startPushButton->setText("Stop");
        }

        break;

    case RUNNING:
        test(m_currentTestRoutine, false, states);
        ui->startPushButton->setText("Exit");

        m_currentTestState = FINISHED;
        break;

    case FINISHED:
        ui->spinBox->setEnabled(true);
        ui->startPushButton->setText("Start");
        ui->spinBox->setFocus(Qt::TabFocusReason);

        clear();
        printMenu();

        m_currentTestState = IDLE;
        break;

    }

}

void TestWidget::test(int id, bool start, int &states)
{
    switch(id)
    {
    default:
    case 0:
        print("Not yet implemented");
        break;
    case 1:
        test_ReaderInfo(start, states);
        break;
    case 2:
        test_Attenuation(start, states);
        break;
    case 3:
        test_Frequency(start, states);
        break;
    case 4:
        test_Sensitivity(start, states);
        break;
    case 5:
        test_Heartbeat(start, states);
        break;
    case 6:
        test_GPIO_Output(start, states);
        break;
    case 7:
        test_GPIO_Input(start, states);
        break;
    case 8:
        test_Inventory(start, states);
        break;
    case 9:
        test_SingleInventory(start, states);
        break;

    // PUR tests
    case 21:
        test_InventoryRSSI(start, states);
        break;
    case 22:
        test_InventoryPostDetectREAD(start, states);
        break;
    case 23:
        test_ReadTID(start, states);
        break;
    case 24:
        test_ReadWriteUser(start, states);
        break;
    case 25:
        test_ReadBlockWriteUser(start, states);
        break;

    case 31:
        test_AN001_ReadTIDFirstTag_Slow(start, states);
        break;
    case 32:
        test_AN001_ReadTIDFirstTag_Fast(start, states);
        break;
    case 33:
        test_AN001_ReadTIDFirstTag_Handle(start, states);
        break;
    }
}

void TestWidget::test_ReaderInfo(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    bool ok = false;

    // get reader id
    ulong readerId = 0;
    ok = m_ph->getReaderID(readerId);
    if (!ok)
        print("ERROR: Could not get ReaderID");

    // get reader type
    ulong readerType = 0;
    ok = m_ph->getReaderType(readerType);
    if (!ok)
        print("ERROR: Could not get ReaderType");

    // get hardware revision
    ulong hwRev = 0;
    ok = m_ph->getHardwareRevision(hwRev);
    if (!ok)
        print("ERROR: Could not get HardwareRevision");

    // get software revision
    ulong swRev = 0;
    ok = m_ph->getSoftwareRevision(swRev);
    if (!ok)
        print("ERROR: Could not get SoftwareRevision");

    // get bootloader revision
    ulong blRev = 0;
    ok = m_ph->getBootloaderRevision(blRev);
    if (!ok)
        print("ERROR: Could not get BootloaderRevision");

    // get current system
    QString system = 0;
    ok = m_ph->getCurrentSystem(system);
    if (!ok)
        print("ERROR: Could not get CurrentSystem");

    // get current state
    QrfeReaderInterface::eRFE_CURRENT_READER_STATE state = QrfeReaderInterface::RFE_STATE_IDLE;
    ok = m_ph->getCurrentState(state);
    if (!ok)
        print("ERROR: Could not get CurrentState");

    // get status register
    qulonglong statusReg = 0;
    ok = m_ph->getStatusRegister(statusReg);
    if (!ok)
        print("ERROR: Could not get StatusRegister");

    // print out results
    print("Reader Information:");
    print("\t -> ReaderID       = " + QString("%1").arg(readerId, 8, 16, QChar('0')).toLatin1());
    print("\t -> ReaderType     = " + QString("%1").arg(readerType, 8, 16, QChar('0')).toLatin1());
    print("\t -> HardwareRev    = " + QString("%1").arg(hwRev, 8, 16, QChar('0')).toLatin1());
    print("\t -> SoftwareRev    = " + QString("%1").arg(swRev, 8, 16, QChar('0')).toLatin1());
    print("\t -> BootloaderRev  = " + QString("%1").arg(blRev, 8, 16, QChar('0')).toLatin1());
    print("\t -> Current System = " + system.toLatin1());
    print("\t -> Current State  = " + QString("%1").arg(state).toLatin1());
    print("\t -> StatusRegister = " + QString("%1").arg(statusReg, 16, 16, QChar('0')).toLatin1());
}

void TestWidget::test_Attenuation(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    print("Testing attenuation settings:");

    ushort attMax, attCur, attTemp;

    // get attenuation values
    print("\t -> 1, Reading attenuation settings:");
    m_ph->getAttenuation(attMax, attCur);
    print("\t\t Attenuation Max=" + QString::number(attMax) + "Current=" + QString::number(attCur));
    print("");

    // set attenuation to fix value 10
    print("\t -> 2, Setting attenuation settings:");
    m_ph->setAttenuation(10);
    print("\t\t Set Attenuation to 10");
    print("");

    // get attenuation settings and check if the fix value is set
    print("\t -> 3, Reading attenuation settings:");
    m_ph->getAttenuation(attMax, attTemp);
    print("\t\t Attenuation Max=" + QString::number(attMax) + "Current=" + QString::number(attTemp));
    print("\t\t Current attenuation " + QString::number(attTemp) + " == 10?");
    if (attTemp != 10)
    {
        print("ERROR------------------ Set Attenuation is not the Current Attenuation");
        return;
    }
    print("\t\t OK\n");

    // retore attenuation to the previous value
    print("\t -> 4, Restore attenuation settings:");
    m_ph->setAttenuation(attCur);
    print("\t\t Set Attenuation to 0");
    print("");

    // check the set values again
    print("\t -> 5, Reading attenuation settings:");
    m_ph->getAttenuation(attMax, attCur);
    print("\t\t Attenuation Max=" + QString::number(attMax) + "Current=" + QString::number(attCur));
    print("");
}

void TestWidget::test_Frequency(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    print("Testing frequency settings:");

    uchar mode, maxCount;
    QList<uint> frequ;
    QList<uint> tempFrequ;

    // get current frequency table
    print("\t -> 1, Reading frequency settings:");
    m_ph->getFrequency(mode, maxCount, frequ);
    print("\t\t FrequencyTable Mode=" + QString::number(mode) + " Max=" + QString::number(maxCount) + " Current=" + QString::number(frequ.size()));
    for (int i = 0; i < frequ.size(); i++)
    {
        print("\t\t\t " + QString::number(i) + " = " + QString::number(frequ[i]));
    }
    print("");

    // create new frequency table and set this
    print("\t -> 2, Setting frequency settings:");
    QList<uint> newFreq;
    newFreq.append(855000);
    newFreq.append(856000);
    newFreq.append(857000);
    m_ph->setFrequency(mode, newFreq);
    print("\t\t Set FrequencyTable to Mode=" + QString::number(mode));
    for (int i = 0; i < newFreq.size(); i++)
    {
        print("\t\t\t " + QString::number(i) + " = " + QString::number(newFreq[i]));
    }
    print("");

    // retrieve the table again and check if the retireved table is the same like the set
    print("\t -> 3, Reading frequency settings:");
    m_ph->getFrequency( mode,  maxCount,  tempFrequ);
    print("\t\t FrequencyTable Mode=" + QString::number(mode) + " Max=" + QString::number(maxCount) + " Current=" + QString::number(tempFrequ.size()));
    for (int i = 0; i < tempFrequ.size(); i++)
    {
        print("\t\t\t " + QString::number(i) + " = " + QString::number(tempFrequ[i]));
    }
    print("\t\t Last set Frequency Table == Read Frequency Table?");
    if (newFreq.toSet() != tempFrequ.toSet())
    {
        print("ERROR------------------ Previous Set Frequency Table is not the Current Frequency Table!!");
        return;
    }
    print("\t\t OK\n");

    // restore the previous frequency table
    print("\t -> 4, Restoring frequency settings:");
    m_ph->setFrequency(mode, frequ);
    print("\t\t Set FrequencyTable to Mode=" + QString::number(mode));
    for (int i = 0; i < frequ.size(); i++)
    {
        print("\t\t\t " + QString::number(i) + " = " + QString::number(frequ[i]));
    }
    print("");

    // recheck the frequency table
    print("\t 5, Reading frequency settings:");
    m_ph->getFrequency( mode,  maxCount,  tempFrequ);
    print("\t\t FrequencyTable Mode=" + QString::number(mode) + " Max=" + QString::number(maxCount) + " Current=" + QString::number(tempFrequ.size()));
    for (int i = 0; i < tempFrequ.size(); i++)
    {
        print("\t\t\t " + QString::number(i) + " = " + QString::number(tempFrequ[i]));
    }
    print("");
}

void TestWidget::test_Sensitivity(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    print("Testing sensitivity settings:");

    short maxSens, minSens, curSens, temSens, actSens;

    // get current sensitivity
    print("\t -> 1, Reading sensitivity settings:");
    m_ph->getSensitivity( maxSens,  minSens,  curSens);
    print("\t\t Sensitivity Max=" + QString::number(maxSens) + " Min=" + QString::number(minSens) + " Current=" + QString::number(curSens));
    print("");

    // set minimum sensitivity and read out the actual set value
    print("\t -> 2, Setting sensitivity settings to MIN:");
    m_ph->setSensitivity(minSens,  actSens);
    print("\t\t Sensitivity Min=" + QString::number(minSens) + " Actual=" + QString::number(actSens));
    print("");

    // retrieve the current sensitivity and check if the current value is the same like the actual set one
    print("\t -> 3, Reading sensitivity settings:");
    m_ph->getSensitivity( maxSens,  minSens,  temSens);
    print("\t\t Sensitivity Max=" + QString::number(maxSens) + " Min=" + QString::number(minSens) + " Current=" + QString::number(temSens));
    print("\t\t Current Sensitivity " + QString::number(temSens) + " == " + QString::number(actSens) + " Actual Set Sensitivirty?");
    if (actSens != temSens)
    {
        print("ERROR------------------ Actual Set Sensitivity is not the Current Sensitivity");
        return;
    }
    print("\t\t OK\n");

    // set maximum sensitivity and read out the actual set value
    print("\t -> 4, Setting sensitivity settings to MAX:");
    m_ph->setSensitivity(maxSens,  actSens);
    print("\t\t Sensitivity Max=" + QString::number(maxSens) + " Actual=" + QString::number(actSens));
    print("");


    // retrieve the current sensitivity and check if the current value is the same like the actual set one
    print("\t -> 5, Reading sensitivity settings:");
    m_ph->getSensitivity( maxSens,  minSens,  temSens);
    print("\t\t Sensitivity Max=" + QString::number(maxSens) + " Min=" + QString::number(minSens) + " Current=" + QString::number(temSens));
    print("\t\t Current Sensitivity " + QString::number(temSens) + " == " + QString::number(actSens) + " Actual Set Sensitivirty?");
    if (actSens != temSens)
    {
        print("ERROR------------------ Actual Set Sensitivity is not the Current Sensitivity");
        return;
    }
    print("\t\t OK\n");

    // restore the previous set value
    print("\t -> 6, Restore sensitivity settings:");
    m_ph->setSensitivity(curSens,  actSens);
    print("\t\t Sensitivity Cur=" + QString::number(curSens) + " Actual=" + QString::number(actSens));
    print("\t\t Current Sensitivity " + QString::number(curSens) + " == " + QString::number(actSens) + " Actual Set Sensitivirty?");
    if (actSens != curSens)
    {
        print("ERROR------------------ Actual Set Sensitivity is not the Current Sensitivity");
        return;
    }
    print("\t\t OK\n");
}

void TestWidget::test_Heartbeat(bool start, int &states)
{
    states = 2;

    if(start)
    {
        print("Testing Heartbeat:");

        // turn on heartbeat with an interval of 250ms
        print("\t -> 1, Setting Heartbeat ON with interval 250ms");
        m_heartBeatTime.start();
        m_ph->setHeartBeat(QrfeReaderInterface::HEARTBEAT_ON, 250);
    }
    else
    {
        // turn off heartbeat
        print("\t -> 2, Setting Heartbeat OFF");
        m_ph->setHeartBeat(QrfeReaderInterface::HEARTBEAT_OFF, 250);
    }
}

void TestWidget::test_GPIO_Output(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    print("Test GPIO - Output");

    ulong mask, output, input;
    if (!m_ph->getGPIOCaps(mask, output, input))
    {
        print("ERROR: Could not get GPIO Caps");
        return;
    }

    print("Setting each pin output.");
    if (!m_ph->setGPIODirection(output))
    {
        print("ERROR: Could not set GPIO direction");
        return;
    }

    ulong test_mask = 0;
    for (int i = 0; i < 32; i++)
    {
        test_mask = (ulong)(((int)1) << i);
        if ((test_mask & output) == 0)
            continue;

        print("Setting pin: " + QString("0x%1").arg(test_mask, 8, 16, QChar('0')));
        if (!m_ph->setGPIO(test_mask))
        {
           print("ERROR: Could not set GPIO");
            break;
        }

        Sleeper::MSleep(1000);

        if (!m_ph->clearGPIO(test_mask))
        {
            print("ERROR: Could not clear GPIO");
            break;
        }
    }

    print("Setting all Pins low.");
    if (!m_ph->clearGPIO(mask))
    {
        print("ERROR: Could not clear GPIO");
        return;
    }
}

void TestWidget::test_GPIO_Input(bool start, int &states)
{
    states = 2;

    if(start)
    {
        print("Test GPIO - Input");

        ulong mask, output, input;
        if (!m_ph->getGPIOCaps(mask, output, input))
        {
            print("ERROR: Could not get GPIO Caps");
            return;
        }

        print("Setting each input pin as input: " + QString("0x%1").arg(input, 8, 16, QChar('0')));
        if (!m_ph->setGPIODirection(~input))
        {
            print("ERROR: Could not set GPIO direction");
            return;
        }

        print();
        print("Press \"Stop\" to clear the Pin and to proceed");
    }
    else
    {

        ulong mask, output, input;
        if (!m_ph->getGPIOCaps(mask, output, input))
        {
            print("ERROR: Could not get GPIO Caps");
            return;
        }

        print("Setting all pins output.");
        if (!m_ph->setGPIODirection(input))
        {
            print("ERROR: Set all input pins to output");
            return;
        }
    }
}

void TestWidget::test_Inventory(bool start, int &states)
{
    states = 2;

    if(start)
    {
        print("Testing Cyclic Inventory:");

        // turn on cyclic inventory
        print("\t -> 1, Starting Cyclic Inventory (Press Enter to Stop)");
        m_cyclicInvCount = 0;
        m_cyclicInvTime.start();
        m_ph->setCyclicInventory(true);
    }
    else
    {
        // turn off cyclic inventory and calculate read rate
        m_ph->setCyclicInventory(false);
        double readRate = (double)m_cyclicInvCount / ((double)m_cyclicInvTime.elapsed()/1000.0);
        print("\t -> 2, Stopped Cyclic Inventry with a ReadRate of " + QString::number(readRate) + " reads/sec");
    }
}

void TestWidget::test_SingleInventory(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    // do a single inventory and print out the list
    print("Do SingleInventory");
    QList<QrfeReaderInterface::QrfeTagEvent> tagList;
    m_ph->doSingleInventory(tagList);
    print("\t\t -> Found " + QString::number(tagList.size()) + " Tags");

    QStringList result;
    foreach (QrfeReaderInterface::QrfeTagEvent tag, tagList)
    {
        result << tag.toString();
    }

    result.sort();
    uchar i = 1;
    foreach (QString str, result)
        print("\t\t #" + QString("%1").arg(i++, 2) + " " + str);
}


void TestWidget::test_InventoryRSSI(bool start, int &states)
{
    states = 2;

    if(start)
    {
        QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
        if (pur_ph == 0)
        {
            states = 1;
            print("Unable to test. Protocol implementation is not valid. Should be PUR.");
            return;
        }

        print("Testing Cyclic Inventory with RSSI:");

        // switch sending of rssi on for PUR reader
        print("\t -> 1, Switching RSSI On");
        pur_ph->setParam(0x02, QByteArray(1, (char)1));

        // turn on cyclic inventory
        print("\t -> 2, Starting Cyclic Inventory (Press Enter to Stop)");
        m_cyclicInvCount = 0;
        m_cyclicInvTime.start();
        pur_ph->setCyclicInventory(true);
    }
    else
    {
        QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
        if (pur_ph == 0)
        {
            print("Unable to test. Protocol implementation is not valid. Should be PUR.");
            return;
        }

        // turn off cyclic inventory and calculate read rate
        pur_ph->setCyclicInventory(false);
        double readRate = (double)m_cyclicInvCount / ((double)m_cyclicInvTime.elapsed()/1000.0);
        print("\t -> 3, Stopped Cyclic Inventry with a ReadRate of " + QString::number(readRate) + " reads/sec");

        // switch sending of rssi off
        print("\t -> 4, Switching RSSI On");
        pur_ph->setParam(0x02, QByteArray(1, (char)0));
    }
}

void TestWidget::test_InventoryPostDetectREAD(bool start, int &states)
{
    states = 2;

    if(start)
    {
        QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
        if (pur_ph == 0)
        {
            states = 1;
            print("Unable to test. Protocol implementation is not valid. Should be PUR.");
            return;
        }

        print("Testing Cyclic Inventory with PostDetect-READ:");

        // switch postdetect-READ on
        print("\t -> 1, Switching PostDetect-READ On");
        pur_ph->setPostDetectReadParams(true, 0x02, 0x0000, 0x04, QByteArray(4, (char)0));

        // turn on cyclic inventory
        print("\t -> 2, Starting Cyclic Inventory (Press Enter to Stop)");
        m_cyclicInvCount = 0;
        m_cyclicInvTime.start();
        pur_ph->setCyclicInventory(true);
    }
    else
    {
        QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
        if (pur_ph == 0)
        {
            print("Unable to test. Protocol implementation is not valid. Should be PUR.");
            return;
        }

        // turn off cyclic inventory and calculate read rate
        pur_ph->setCyclicInventory(false);
        double readRate = (double)m_cyclicInvCount / ((double)m_cyclicInvTime.elapsed()/1000.0);
        print("\t -> 3, Stopped Cyclic Inventry with a ReadRate of " + QString::number(readRate) + " reads/sec");

        // switch postdetect-READ off
        print("\t -> 4, Switching PostDetect-READ Off");
        pur_ph->setPostDetectReadParams(false, 0x02, 0x0000, 0x04, QByteArray(4, (char)0));
    }
}

void TestWidget::test_ReadTID(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
    if (pur_ph == 0)
    {
        print("Unable to test. Protocol implementation is not valid. Should be PUR.");
        return;
    }

    print("Trying to read tid register:");

    // do a single inventory and print out the tag list with an index to select one tag
    print("\t -> 1, Searching for tags:");
    QList<QrfeReaderInterface::QrfeTagEvent> tagList;
    pur_ph->doSingleInventory(tagList);
    print("\t\t -> Found " + QString::number(tagList.size()) + " Tags");

    print("");
    print("\t -> 2, Select a tag by index: > ");
    QByteArray selectedEpc;
    if (tagList.size() > 0)
    {
        QStringList tags;
        foreach (QrfeReaderInterface::QrfeTagEvent tag, tagList)
            tags << tag.tagId.toHex();

        bool ok = false;
        QString tag = QInputDialog::getItem(this, "Select tag", "Select tag:", tags, 0, false, &ok);
        if(!ok || tag.isNull())
            return;

        selectedEpc = QByteArray::fromHex(tag.toLatin1());
    }
    else
    {
        print("\t\t Found 0 Tags, stopping the test...");
        return;
    }

    print("");
    // try to read the first 4 bytes of the TID membank of the selected tag
    print("\t -> 3, Trying to read first 4 bytes of TID register:");
    QByteArray passwd(4, (char)0);
    QByteArray data;
    if (pur_ph->readFromTag(selectedEpc, 0x02, 0, passwd, 4, data))
    {
        print("\t\t Read succeeded, read data: " + data.toHex());
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
    }
    print("");
}

void TestWidget::test_ReadWriteUser(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
    if (pur_ph == 0)
    {
        print("Unable to test. Protocol implementation is not valid. Should be PUR.");
        return;
    }

    print("Trying to read write user register:");

    // do a single inventory and print out the tag list with an index to select one tag
    print("\t -> 1, Searching for tags:");
    QList<QrfeReaderInterface::QrfeTagEvent> tagList;
    pur_ph->doSingleInventory(tagList);
    print("\t\t -> Found " + QString::number(tagList.size()) + " Tags");

    print("");
    print("\t -> 2, Select a tag by index: > ");
    QByteArray selectedEpc;
    if (tagList.size() > 0)
    {
        QStringList tags;
        foreach (QrfeReaderInterface::QrfeTagEvent tag, tagList)
            tags << tag.tagId.toHex();

        bool ok = false;
        QString tag = QInputDialog::getItem(this, "Select tag", "Select tag:", tags, 0, false, &ok);
        if(!ok || tag.isNull())
            return;

        selectedEpc = QByteArray::fromHex(tag.toLatin1());
    }
    else
    {
        print("\t\t Found 0 Tags, stopping the test...");
        return;
    }

    uchar memBank = 0x03;
    QByteArray passwd(4, (char)0);
    QByteArray userMemBefore;
    QByteArray userMemAfter;
    QByteArray old4Bytes;
    QByteArray new4Bytes;

    new4Bytes.append(0x11);
    new4Bytes.append(0x22);
    new4Bytes.append(0x33);
    new4Bytes.append(0x44);

    // try to read the whole user memory bank
    print("\t -> 3, Trying to Read all bytes from user mem");
    if (pur_ph->readFromTag(selectedEpc, memBank, 0, passwd, 0, userMemBefore))
    {
        print("\t\t Read succeeded, read data:");
        QString mem;
        for (int i = 0; i < userMemBefore.size(); i++)
        {
            mem += (QString("%1-").arg(userMemBefore[i], 2, 16, QChar('0')));
            if(((i + 1)%8) == 0)
                mem += ("\n\t\t\t ");
        }
        print("\t\t\t " + mem);
        print("\n");
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // save the old first 4 bytes of the user memory
    old4Bytes = userMemBefore.left(4);

    // try to write 4 dummy bytes to the user mem
    print("\t -> 4, Trying to Wrtie data:");
    print("\t\t\t " + new4Bytes.toHex());
    if (pur_ph->writeToTag(selectedEpc, memBank, 0, passwd, new4Bytes))
    {
        print("\t\t Wrtie succeeded");
    }
    else
    {
        print("\t\t Write failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // try to read the whole user me again
    print("\t -> 5, Trying to Read written data");
    if (pur_ph->readFromTag(selectedEpc, memBank, 0, passwd, 0, userMemAfter))
    {
        print("\t\t Read succeeded, read data:");
        QString mem;
        for (int i = 0; i < userMemAfter.size(); i++)
        {
            mem += (QString("%1-").arg(userMemAfter[i], 2, 16, QChar('0')));
            if(((i + 1)%8) == 0)
                mem += ("\n\t\t\t ");
        }
        print("\t\t\t " + mem);
        print("\n");
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // try to write the old bytes into the memory
    print("\t -> 6, Trying to restore the bytes");
    if (pur_ph->writeToTag(selectedEpc, memBank, 0, passwd, old4Bytes))
    {
        print("\t\t Wrtie succeeded");
    }
    else
    {
        print("\t\t Write failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // again read the restored bytes from the tag
    print("\t -> 7, Trying to Read written data");
    if (pur_ph->readFromTag(selectedEpc, memBank, 0, passwd, 0, userMemAfter))
    {
        print("\t\t Read succeeded, read data:");
        QString mem;
        for (int i = 0; i < userMemAfter.size(); i++)
        {
            mem += (QString("%1-").arg(userMemAfter[i], 2, 16, QChar('0')));
            if(((i + 1)%8) == 0)
                mem += ("\n\t\t\t ");
        }
        print("\t\t\t " + mem);
        print("\n");
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");
}

void TestWidget::test_ReadBlockWriteUser(bool start, int &states)
{
    states = 1;

    uchar blockSize = 2;

    if(!start)
        return;

    QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
    if (pur_ph == 0)
    {
        print("Unable to test. Protocol implementation is not valid. Should be PUR.");
        return;
    }

    print("Trying to read write user register:");

    // do a single inventory and print out the tag list with an index to select one tag
    print("\t -> 1, Searching for tags:");
    QList<QrfeReaderInterface::QrfeTagEvent> tagList;
    pur_ph->doSingleInventory(tagList);
    print("\t\t -> Found " + QString::number(tagList.size()) + " Tags");

    print("");
    print("\t -> 2, Select a tag by index: > ");
    QByteArray selectedEpc;
    if (tagList.size() > 0)
    {
        QStringList tags;
        foreach (QrfeReaderInterface::QrfeTagEvent tag, tagList)
            tags << tag.tagId.toHex();

        bool ok = false;
        QString tag = QInputDialog::getItem(this, "Select tag", "Select tag:", tags, 0, false, &ok);
        if(!ok || tag.isNull())
            return;

        selectedEpc = QByteArray::fromHex(tag.toLatin1());
    }
    else
    {
        print("\t\t Found 0 Tags, stopping the test...");
        return;
    }

    uchar memBank = 0x03;
    QByteArray passwd(4, (char)0);
    QByteArray userMemBefore;
    QByteArray userMemAfter;
    QByteArray old4Bytes;
    QByteArray new4Bytes;

    new4Bytes.append(0x11);
    new4Bytes.append(0x22);
    new4Bytes.append(0x33);
    new4Bytes.append(0x44);

    // try to read the whole user memory bank
    print("\t -> 3, Trying to Read all bytes from user mem");
    if (pur_ph->readFromTag(selectedEpc, memBank, 0, passwd, 0, userMemBefore))
    {
        print("\t\t Read succeeded, read data:");
        QString mem;
        for (int i = 0; i < userMemBefore.size(); i++)
        {
            mem += (QString("%1-").arg(userMemBefore[i], 2, 16, QChar('0')));
            if(((i + 1)%8) == 0)
                mem += ("\n\t\t\t ");
        }
        print("\t\t\t " + mem);
        print("\n");
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // save the old first 4 bytes of the user memory
    old4Bytes = userMemBefore.left(4);

    // try to write 4 dummy bytes to the user mem
    print("\t -> 4, Trying to Wrtie data:");
    print("\t\t\t " + new4Bytes.toHex());
    if (pur_ph->blockWriteToTag(selectedEpc, memBank, 0, passwd, blockSize, new4Bytes))
    {
        print("\t\t Wrtie succeeded");
    }
    else
    {
        print("\t\t Write failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // try to read the whole user me again
    print("\t -> 5, Trying to Read written data");
    if (pur_ph->readFromTag(selectedEpc, memBank, 0, passwd, 0, userMemAfter))
    {
        print("\t\t Read succeeded, read data:");
        QString mem;
        for (int i = 0; i < userMemAfter.size(); i++)
        {
            mem += (QString("%1-").arg(userMemAfter[i], 2, 16, QChar('0')));
            if(((i + 1)%8) == 0)
                mem += ("\n\t\t\t ");
        }
        print("\t\t\t " + mem);
        print("\n");
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // try to write the old bytes into the memory
    print("\t -> 6, Trying to restore the bytes");
    if (pur_ph->blockWriteToTag(selectedEpc, memBank, 0, passwd, blockSize, old4Bytes))
    {
        print("\t\t Wrtie succeeded");
    }
    else
    {
        print("\t\t Write failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");

    // again read the restored bytes from the tag
    print("\t -> 7, Trying to Read written data");
    if (pur_ph->readFromTag(selectedEpc, memBank, 0, passwd, 0, userMemAfter))
    {
        print("\t\t Read succeeded, read data:");
        QString mem;
        for (int i = 0; i < userMemAfter.size(); i++)
        {
            mem += (QString("%1-").arg(userMemAfter[i], 2, 16, QChar('0')));
            if(((i + 1)%8) == 0)
                mem += ("\n\t\t\t ");
        }
        print("\t\t\t " + mem);
        print("\n");
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
        return;
    }
    print("");
}


void TestWidget::test_AN001_ReadTIDFirstTag_Slow(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
    if (pur_ph == 0)
    {
        print("Unable to test. Protocol implementation is not valid. Should be PUR.");
        return;
    }

    print("Trying to read tid register:");


    print("\t -> 1, Searching for tags:");

    m_cyclicInvCount = 0;
    m_lastTagEventValid = false;
    QTime startTime;

    pur_ph->setCyclicInventory(true);
    bool res = waitForSignal(pur_ph, SIGNAL(cyclicInventory(QrfeReaderInterface::QrfeTagEvent)), 5000);
    pur_ph->setCyclicInventory(false);
    startTime.start();

    if(!res)
    {
        print("\t\t Did not find any tag...");
        return;
    }

    print("\t\t Found Tag " + m_lastTagEvent.tagId.toHex());

    // try to read the first 4 bytes of the TID membank of the detected tag
    print("\t -> 2, Trying to read first 4 bytes of TID register:");
    QByteArray passwd(4, (char)0);
    QByteArray data;
    if (pur_ph->readFromTag(m_lastTagEvent.tagId, 0x02, 0, passwd, 4, data))
    {
        print("\t\t Read succeeded, read data: " + data.toHex());
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
    }
    print("");
    print("Read Result " + QString::number(startTime.elapsed()) + "ms after the tag was detected");
    print("");
}

void TestWidget::test_AN001_ReadTIDFirstTag_Fast(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
    if (pur_ph == 0)
    {
        print("Unable to test. Protocol implementation is not valid. Should be PUR.");
        return;
    }

    print("Trying to read tid register:");


    print("\t -> 1, Searching for tags:");

    m_cyclicInvCount = 0;
    m_lastTagEventValid = false;
    QTime startTime;

    pur_ph->setCyclicInventory(true);
    bool res = waitForSignal(pur_ph, SIGNAL(cyclicInventory(QrfeReaderInterface::QrfeTagEvent)), 5000);
    pur_ph->setBlockCyclicInventoryInterrupts(true);
    startTime.start();

    if(!res)
    {
        pur_ph->setCyclicInventory(false);
        pur_ph->setBlockCyclicInventoryInterrupts(false);
        print("\t\t Did not find any tag...");
        return;
    }

    print("\t\t Found Tag " + m_lastTagEvent.tagId.toHex());

    // try to read the first 4 bytes of the TID membank of the detected tag
    print("\t -> 2, Trying to read first 4 bytes of TID register:");
    QByteArray passwd(4, (char)0);
    QByteArray data;
    if (pur_ph->readFromTag(m_lastTagEvent.tagId, 0x02, 0, passwd, 4, data))
    {
        print("\t\t Read succeeded, read data: " + data.toHex());
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
    }
    print("");
    print("Read Result " + QString::number(startTime.elapsed()) + "ms after the tag was detected");
    print("");

    pur_ph->setCyclicInventory(false);
    pur_ph->setBlockCyclicInventoryInterrupts(false);
}

void TestWidget::test_AN001_ReadTIDFirstTag_Handle(bool start, int &states)
{
    states = 1;

    if(!start)
        return;

    QrfeReaderInterface::QrfePURprotocolHandler* pur_ph = dynamic_cast<QrfeReaderInterface::QrfePURprotocolHandler*>(m_ph);
    if (pur_ph == 0)
    {
        print("Unable to test. Protocol implementation is not valid. Should be PUR.");
        return;
    }

    print("Trying to read tid register:");

    pur_ph->setParam(0x24, QByteArray(1, (char) 0x01));        //send Handle ON
    pur_ph->setParam(0x00, QByteArray(1, (char) 0x04));        //inventory_secure_single_tag
    pur_ph->setParam(0x03, QByteArray(1, (char) 0x02));        //sendTagIDImmediatelyAndStop

    print("\t -> 1, Searching for tags:");

    m_cyclicInvCount = 0;
    m_lastTagEventValid = false;
    QTime startTime;

    pur_ph->setAntennaPower(true);
    pur_ph->setCyclicInventory(true);
    bool res = waitForSignal(pur_ph, SIGNAL(cyclicInventory(QrfeReaderInterface::QrfeTagEvent)), 5000);
    startTime.start();

    if(!res)
    {
        print("\t\t Did not find any tag...");

        pur_ph->setCyclicInventory(false);
        pur_ph->setAntennaPower(false);
        pur_ph->setParam(0x24, QByteArray(1, (char) 0x00));        //send Handle OFF
        pur_ph->setParam(0x00, QByteArray(1, (char) 0x02));        //inventory_secure_multi_tag
        pur_ph->setParam(0x03, QByteArray(1, (char) 0x00));        //sendTagIDImmediately
        return;
    }

    print("\t\t Found Tag " + m_lastTagEvent.tagId.toHex());

    // try to read the first 4 bytes of the TID membank of the detected tag
    print("\t -> 2, Trying to read first 4 bytes of TID register:");
    QByteArray passwd(4, (char)0);
    QByteArray data;
    if (pur_ph->readFromTag(m_lastTagEvent.tagId, 0x02, 0, passwd, 4, data))
    {
        print("\t\t Read succeeded, read data: " + data.toHex());
    }
    else
    {
        print("\t\t Read failed:" + QrfeReaderInterface::toString(pur_ph->getLastReturnCode()));
    }
    print("");
    print("Read Result " + QString::number(startTime.elapsed()) + "ms after the tag was detected");
    print("");

    pur_ph->setCyclicInventory(false);
    pur_ph->setAntennaPower(false);
    pur_ph->setParam(0x24, QByteArray(1, (char) 0x00));        //send Handle OFF
    pur_ph->setParam(0x00, QByteArray(1, (char) 0x02));        //inventory_secure_multi_tag
    pur_ph->setParam(0x03, QByteArray(1, (char) 0x00));        //sendTagIDImmediately
}


void TestWidget::heartBeat ()
{
    uint elapsed = m_heartBeatTime.elapsed();
    m_heartBeatTime.start();
    print("[E] Heartbeat after " + QString::number(elapsed));
}

void TestWidget::cyclicInventory (const QrfeReaderInterface::QrfeTagEvent &tagInfo)
{
    if(!m_lastTagEventValid)
    {
        m_lastTagEventValid = true;
        m_lastTagEvent = tagInfo;
    }

    QString info;
    info = "\t[E] " + QString("%1").arg(++m_cyclicInvCount, 8, 10, QChar(' ')) + " " + tagInfo.tagId.toHex() + " ";

    if (tagInfo.hasMemory)
    {
        info += " MEM@" + QString::number(tagInfo.memBank) + "." + QString::number(tagInfo.memAddr) + ":" + tagInfo.memData.toHex() + " ";
    }

    if (tagInfo.hasApplicationInfo)
    {
        info += " APP:" + tagInfo.applicationInfo.toHex() + " ";
    }

    print(info);

    if (tagInfo.hasRSSI)
    {
        QString q_str;
        q_str = "\t\t\tQ: ";
        for (int i = 0; i < tagInfo.rssi[0]; i++)
            q_str.append("#");
        print(q_str);

        QString i_str;
        i_str = "\t\t\tI: ";
        for (int i = 0; i < tagInfo.rssi[1]; i++)
            i_str.append("#");
        print(i_str);
    }
}

void TestWidget::stateChanged (const int newState)
{
    print("[E] State changed to: " + QrfeReaderInterface::toString((QrfeReaderInterface::eRFE_CURRENT_READER_STATE)newState));
}

void TestWidget::statusRegisterChanged (const qulonglong statusRegister)
{
    print("[E] Status register changed " + QString("0x%1").arg(statusRegister, 16, 16, QChar('0')));
}

void TestWidget::gpioValuesChanged ( const ulong gpioValues )
{
    print("[E] GPIO values changed " + QString("0x%1").arg(gpioValues, 8, 16, QChar('0')));
}



void TestWidget::printMenu()
{
    clear();
    print("+------------------------ MENU ----------------------------+");
    print("|    1 = Query Reader Information                          |");
    print("|    2 = Test Attenuation Settings                         |");
    print("|    3 = Test Frequency Settings                           |");
    print("|    4 = Test Sensitivity Settings                         |");
    print("|    5 = Test Heartbeat                                    |");
    print("|    6 = Test GPIO - Output                                |");
    print("|    7 = Test GPIO - Input                                 |");
    print("|    8 = Start Inventory                                   |");
    print("|    9 = Do SinglInventory                                 |");
    print("+-PUR------------------------------------------------------+");
    print("|   21 = Start RSSI Inventory                              |");
    print("|   22 = Start PostDetect READ Inventory                   |");
    print("|   23 = Try to Read TID                                   |");
    print("|   24 = Try to Read/Write User Mem                        |");
    print("|   25 = Try to Read/BlockWrite User Mem                   |");
    print("+-AN-PUR---------------------------------------------------+");
    print("|   31 = AN001 - PUR - Test Read TID of First Tag - Slow   |");
    print("|   32 = AN001 - PUR - Test Read TID of First Tag - Fast   |");
    print("|   33 = AN001 - PUR - Test Read TID of First Tag - Handle |");
    print("+----------------------------------------------------------+");
    print("");
    print("Please choose: > ");
}

void TestWidget::print(const QString& str)
{
    ui->textBrowser->append(str);
    qApp->processEvents();
}

void TestWidget::clear()
{
    ui->textBrowser->clear();
    qApp->processEvents();
}


bool TestWidget::waitForSignal(QObject *sender, const char *signal, int timeout)
{
    QEventLoop loop;
    QTimer timer;
    timer.setInterval(timeout);
    timer.setSingleShot(true);

    loop.connect(sender, signal, SLOT(quit()));
    loop.connect(&timer, SIGNAL(timeout()), SLOT(quit()));
    timer.start();
    loop.exec();

    return timer.isActive();
}



